//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_ListenerContainer_h)
#define CANDERA_ListenerContainer_h

#include <Candera/Environment.h>
#include <FeatStd/Container/Vector.h>

namespace Candera {
    namespace Internal {
/** @addtogroup CommonBase
 *  @{
 */

/** @brief A container that provides the functionality required by listeners.
 *
 *  Properties:
 *   - possibility to add elements, remove elements and non-destructively
 *          iterate over elements.
 *   - addition, removal and iteration may be called within iteration itself.
 *   - management of elements lifetime is outside the scope of this class;
 *          container might not relieve references as soon as remove is called.
 *   - elements added during an iteration are not touched by the current
 *          iteration.
 *   - the order in which elements are iterated is undefined.
 *   - the same element may be added multiple times; removal only removes one
 *          instance; iteration hits all added instances.
 *
 *  @tparam TListener type of the attached listeners.
 */
template <typename TListener>
class ListenerContainer
{
public:
    typedef TListener Listener;

    /**
     * @brief Class used by the container to notify the listeners
     * during iteration.
     */
    class Event
    {
    public:
        /**
         * Function called on each listener within the container.
         * @param listener Listener which will be notified about the event.
         */
        virtual void Notify(Listener*) = 0;
    };

    /**
     * Constructor.
     */
    ListenerContainer();

    /**
     * Add a listener to the container. See class description for restrictions
     * on lifetime of the listeners.
     * Complexity is linear in worst case, but may be constant in average case.
     * @param listener Listener to add to the container.
     * @return True if memory allocation required to complete the operation
                is successful.
     */
    bool Add(Listener* listener);
    /**
     * Remove a listener from the container. See class description for
     * concrete behavior on iteration and when multiple instances of the same
     * listener are present.
     * Complexity is linear.
     * @param listener Listener to be removed from the container.
     * @return True if any listener was found for removal.
     */
    bool Remove(Listener* listener);
    /**
     * Iterate over the listeners and trigger the event for each listener.
     * See class description for information on iteration order.
     * @param event Event triggered for each of the listeners.
     */
    void Iterate(Event& event);
    /**
     * Verifies whether a listener is within the container.
     * Complexity is linear.
     * @param listener Listener being searched for.
     * @return True if listener was found.
     */
    bool Contains(Listener* listener);
    /**
     * Returns the number of listeners in the container.
     * @return The number of listeners in the container.
     */
    SizeType Size() { return m_container.Size(); }
    /**
     * Remove all listeners and deallocate memory associated with the container.
     */
    void Free() { m_container.Free(); }

private:
    struct Iterator {
        OffsetType m_position;
        OffsetType m_size;

        Iterator() : m_position(0), m_size(0) {}
    };
    typedef FeatStd::Internal::LinearIncreasePolicy<1> IncreasePolicy;
    typedef FeatStd::Internal::Vector<Listener*, IncreasePolicy> Container;
    typedef FeatStd::Internal::Vector<Iterator, IncreasePolicy> IteratorStack;

    Container m_container;
    IteratorStack m_iterators;
};
/// @}


template <typename TListener>
ListenerContainer<TListener>::ListenerContainer()
{
}

template <typename TListener>
bool ListenerContainer<TListener>::Add(Listener* listener)
{
    if (m_container.GetCapacity() == 0) {
        if (!m_container.Reserve(1)) {
            return false;
        }
    }
    // This does not affect the iterator of the container.
    // Current iterations will avoid touching these listeners.
    return m_container.Add(listener);
}

template <typename TListener>
bool ListenerContainer<TListener>::Remove(Listener* listener)
{
    // Iterate from back to front. This should be most efficient for
    // listeners that are added and removed frequently.
    const OffsetType listenerCount = m_container.Size();
    for (OffsetType listenerIndex = listenerCount - 1; listenerIndex >= 0; --listenerIndex) {
        if (m_container[listenerIndex] == listener) {
            // Update iterations that are already in progress.
            OffsetType iteratorCount = m_iterators.Size();
            for (OffsetType iteratorIndex = 0; iteratorIndex < iteratorCount; ++iteratorIndex) {
                // Item removed from the ones already iterated over.
                if (m_iterators[iteratorIndex].m_position <= listenerIndex) {
                    --m_iterators[iteratorIndex].m_position;
                }
                // Item removed from the ones scheduled for iteration.
                if (m_iterators[iteratorIndex].m_size > listenerIndex) {
                    --m_iterators[iteratorIndex].m_size;
                }
            }
            return m_container.Remove(listenerIndex);
        }
    }
    return false;
}

template <typename TListener>
void ListenerContainer<TListener>::Iterate(Event& event)
{
    if (m_container.Size() != 0) {
        if (0 == m_iterators.GetCapacity()) {
            if (!m_iterators.Reserve(1)) {
                return;
            }
        }

        if (!m_iterators.Add(Iterator())) {
            return;
        }
        OffsetType it = m_iterators.Size() - 1;
        // Iteration will never go pass the initial size.
        m_iterators[it].m_size = m_container.Size();
        while(m_iterators[it].m_position < m_iterators[it].m_size) {
            event.Notify(m_container[m_iterators[it].m_position]);
            ++m_iterators[it].m_position;
        }
        m_iterators.Remove(it);

    }
}

template <typename TListener>
bool ListenerContainer<TListener>::Contains(Listener* listener)
{
    // Investigate most recently added elements first.
    return m_container.Contains(listener);
}

    }
}

#endif // CANDERA_ListenerContainer_h
