//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_INSTANCEID_H)
#define CANDERA_INSTANCEID_H

#include <Candera/System/Container/Vector.h>
#include <CanderaPlatform/OS/CanderaTypes.h>
#include <FeatStd/Util/StaticObject.h>

/**
 * InstanceId assigns every instance of type T a unique identification
 * number that is never bigger than the total number of objects of type T.
 * Be careful when caching Ids, because just like pointers, the same Id might
 * be reassigned to a different object when destroying and creating objects
 * of the same type.
 */

namespace Candera
{

namespace Internal
{

template<typename T>
class InstanceId
{
public:
    InstanceId();
    ~InstanceId();

    inline operator Int () const;
    inline static Int GetMaxId();
    static void CleanUp();

private:
    static void SiftUp(SizeType nodeIndex);
    static void SiftDown(SizeType nodeIndex);

    Int m_id;

    // The IdContainer is managed as a heap to always provide the lowest free Id at index 0.
    typedef Candera::Internal::Vector<Int> IdContainer;

    static IdContainer& GetIdContainer() {
        FEATSTD_UNSYNCED_STATIC_OBJECT(IdContainer, s_idContainer);
        return s_idContainer;
    }

    static FeatStd::Internal::CriticalSection& GetCriticalSection() {
        FEATSTD_UNSYNCED_STATIC_OBJECT(FeatStd::Internal::CriticalSection, criticalSection);
        return criticalSection;
    }

    static Int& MaxId() {
        FEATSTD_UNSYNCED_STATIC_OBJECT(Int, s_maxId);
        return s_maxId;
    }

};

template<typename T>
InstanceId<T>::InstanceId()
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    FeatStd::Internal::CriticalSectionLocker lock(&InstanceId<T>::GetCriticalSection());
#endif

    IdContainer& s_freeIds = GetIdContainer();

    if (s_freeIds.Size() > 0) {
        m_id = s_freeIds[0];
        s_freeIds[0] = s_freeIds[s_freeIds.Size() - 1];
        s_freeIds.Remove(s_freeIds.Size() - 1);
        if (s_freeIds.Size() > 0) {
            SiftDown(0);
        }
        else {
            s_freeIds.Free();
        }

        return;
    }

    m_id = ++(MaxId());
}

template<typename T>
InstanceId<T>::~InstanceId()
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    FeatStd::Internal::CriticalSectionLocker lock(&InstanceId<T>::GetCriticalSection());
#endif

    if (m_id != MaxId()) {
        IdContainer& s_freeIds = GetIdContainer();
        s_freeIds.Add(m_id);
        SiftUp(s_freeIds.Size() - 1);
    }
    else {
        --(MaxId());
    }
}

template<typename T>
inline InstanceId<T>::operator Int () const
{
    return m_id;
}

template<typename T>
inline Int InstanceId<T>::GetMaxId()
{
    return MaxId();
}

template<typename T>
void InstanceId<T>::CleanUp()
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    FeatStd::Internal::CriticalSectionLocker lock(&InstanceId<T>::GetCriticalSection());
#endif

    IdContainer& s_freeIds = GetIdContainer();
    bool removedMaxId = true;
    while (removedMaxId) {
        removedMaxId = false;
        const Int maxId = GetMaxId();
        for (SizeType i = 0; i < s_freeIds.Size(); ++i) {
            if (maxId == s_freeIds[i]) {
                s_freeIds.Remove(i);
                --(MaxId());
                removedMaxId = true;
                break;
            }
        }
    }

    if (0 == s_freeIds.Size()) {
        s_freeIds.Free();
    }
}

template<typename T>
void InstanceId<T>::SiftUp(SizeType nodeIndex)
{
    if (nodeIndex != 0) {
        SizeType parentIndex = (nodeIndex - 1) / 2;
        IdContainer& s_freeIds = GetIdContainer();

        if (s_freeIds[parentIndex] > s_freeIds[nodeIndex]) {
            Int tmp = s_freeIds[parentIndex];
            s_freeIds[parentIndex] = s_freeIds[nodeIndex];
            s_freeIds[nodeIndex] = tmp;
            SiftUp(parentIndex);
        }
    }
}

template<typename T>
void InstanceId<T>::SiftDown(SizeType nodeIndex)
{
    SizeType minIndex;
    SizeType leftChildIndex = 2 * nodeIndex + 1;
    SizeType rightChildIndex = leftChildIndex + 1;
    IdContainer& s_freeIds = GetIdContainer();

    if (rightChildIndex >= s_freeIds.Size()) {
        if (leftChildIndex >= s_freeIds.Size()) {
            return;
        }
        else {
            minIndex = leftChildIndex;
        }
    }
    else {
        if (s_freeIds[leftChildIndex] <= s_freeIds[rightChildIndex]) {
            minIndex = leftChildIndex;
        }
        else {
            minIndex = rightChildIndex;
        }
    }

    if (s_freeIds[nodeIndex] > s_freeIds[minIndex]) {
        Int tmp = s_freeIds[minIndex];
        s_freeIds[minIndex] = s_freeIds[nodeIndex];
        s_freeIds[nodeIndex] = tmp;
        SiftDown(minIndex);
    }
}

}

}

#endif
