//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_COLOR_H)
#define CANDERA_COLOR_H

#include <Candera/EngineBase/Common/BaseStringBufferAppenders.h>
#include <Candera/Environment.h>
#include <Candera/System/Mathematics/Vector4.h>
#include <FeatStd/Diagnostics/Debug.h>

namespace Candera {
/** @addtogroup CommonBase
 *  @{
 */

/**
 * @brief Represents a color by its red-, green-, blue- and alpha-value.
 *        Provides color arithmetic functions and operators.
 *        Arithmetic operators on Colors act like on Vectors, results are unclamped,
 *        so user of this class has to call Clamp manually after doing arithmetic operations.
 */
class Color {
    public:

        struct Data {
            Float red;
            Float green;
            Float blue;
            Float alpha;

            Float& operator [] (UInt8 index);
            Float operator [] (UInt8 index) const;

            // Check equality (in contrast to Color::operator== which checks if almost equal)
            bool operator== (const Data& rhs) const { 
                FEATSTD_LINT_NEXT_EXPRESSION(777, CANDERA_LINT_REASON_FLOATCOMPARING)
                return ((red == rhs.red) && (green == rhs.green) && (blue == rhs.blue) && (alpha == rhs.alpha)); }

            bool operator!= (const Data& rhs) const { return (!(*this == rhs)); }
        };

        /**
         * Constructor
         */
        Color();

        /**
         * Destructor
         */
        ~Color(); // non-virtual

        /**
         * Copy Constructor
         * @param color Color to copy from.
         */
        Color(const Color &color);

        /**
         * Constructor
         * @param data The RGBA data with which to create the Color object
         */
        Color(const Data &data);

        /**
         * Constructor
         * @param redValue   value for red in the new color
         * @param greenValue value for green in the new color
         * @param blueValue  value for blue in the new color
         * @param alphaValue value for alpha in the new color, default value fully opaque
         */
        explicit Color(Float redValue, Float greenValue, Float blueValue, Float alphaValue = 1.0F);

        /**
         * Sets new color values
         * @param redValue   value for red in the new color
         * @param greenValue value for green in the new color
         * @param blueValue  value for blue in the new color
         * @param alphaValue value for alpha in the new color, default value fully opaque
         */
        void Set(Float redValue, Float greenValue, Float blueValue, Float alphaValue = 1.0F);

        /**
         * Sets new color values in RGB color space derived from given color values in HSV color space.
         * @param hue        Hue described as angle on a chromatic circle. The angle is given in radians and has to be between
         *                   0.0F and 2 * PI.
         * @param saturation Saturation between 0.0 and 1.0. 0.0 means pure white, 1.0 means pure hue color.
         * @param value      Lightness between 0.0 and 1.0. 0.0F means pure black, 1.0 fully lit color.
         * @param alphaValue Value for alpha in the new color, default value fully opaque.
         */
        void SetHSV(Float hue, Float saturation, Float value, Float alphaValue = 1.0F);

        /**
         * Gets the current color values in RGB color space converted to color values in HSV color space.
         * @param hue        Out parameter: Hue described as angle on a chromatic circle. The angle is given in radians.
         * @param saturation Out parameter: Saturation between 0.0 and 1.0. 0.0 means pure white, 1.0 means pure hue color.
         * @param value      Out parameter: Lightness between 0.0 and 1.0. 0.0F means pure black, 1.0 fully lit color.
         * @param alphaValue Out parameter: Alpha value of this color object.
         */
        void GetHSV(Float& hue, Float& saturation, Float& value, Float& alphaValue) const;

        /**
         * Sets the new red value
         * @param redValue   value to be set for red
         */
        void SetRed(Float redValue);

        /**
         * Gets the color's red value
         * @return the red value
         */
        Float GetRed() const;

        /**
         * Sets the new green value
         * @param greenValue value to be set for green
         */
        void SetGreen(Float greenValue);

        /**
         * Gets the color's green value
         * @return the green value
         */
        Float GetGreen() const;

        /**
         * Sets the new blue blue
         * @param blueValue  value to be set for blue
         */
        void SetBlue(Float blueValue);

        /**
         * Gets the color's blue value
         * @return the blue value
         */
        Float GetBlue() const;

        /**
         * Sets the new alpha value
         * @param alphaValue value to be set for alpha
         */
        void SetAlpha(Float alphaValue);

        /**
         * Gets the color's alpha value
         * @return the alpha value
         */
        Float GetAlpha() const;

        /**
         * Clamps values of all color components to 0.0 and 1.0. Values lower than 0.0 become 0.0.
         * Values higher than 1.0 become 1.0.
         */
        void Clamp();

        /**
         * Inverts RGB values of this clamped color. Alpha is not considered.
         */
        void Invert();

        /**
         * Gets average RGBA color between colors a and b.
         * Average is (a + b) * 0.5
         * @param a First color to get average from.
         * @param b Second color to get average from.
         * @return  Average RGBA color between a and b.
         */
        static Color GetAverageColor(const Color& a, const Color& b);

        /**
         * Gets linear interpolation between colors a and b, weighted with weight.
         * Color lerp = a + (b-a) * weight.
         * @param a First color to be interpolated from.
         * @param b Second color to be interpolated from.
         * @param weight Weight of interpolation. 0.0 means a as result. 1.0 means b as result.
         * @return The interpolated color.
         */
        static Color Lerp(const Color& a, const Color& b, Float weight);

        /**
         * Gets the color's RGBA data
         * @return the Data value
         */
        const Data& GetData() const;

        Color operator+(const Color& rhs) const;
        Color operator-(const Color& rhs) const;
        Color operator*(const Color& rhs) const;

        Color& operator=(const Color& color);
        Color& operator+=(const Color& rhs);
        Color& operator-=(const Color& rhs);
        Color& operator*=(const Color& rhs);

        bool operator==(const Color& color) const;
        bool operator!=(const Color& color) const;

        Color operator+(const Float f) const;
        Color operator-(const Float f) const;
        Color operator*(const Float f) const;

        Color& operator=(const Float f);
        Color& operator+=(const Float f);
        Color& operator-=(const Float f);
        Color& operator*=(const Float f);

        /**
         * Provides access to the properties by specifying an index.
         * Index 0 returns red and accordingly.
         * @param index Index to be resolved. Index has to be in the range 0-3.
         * @return Float value of indexed color.
         */
        Float& operator [] (UInt8 index);
        Float operator [] (UInt8 index) const;

        operator Data&();
        operator const Data&() const;

        Color& operator=(const Data& color);

    private:
        Data m_data;
};

inline void Color::SetRed(Float redValue) { m_data.red = redValue; }
inline Float Color::GetRed() const { return m_data.red; }
inline void Color::SetGreen(Float greenValue) { m_data.green = greenValue; }
inline Float Color::GetGreen() const { return m_data.green; }
inline void Color::SetBlue(Float blueValue) { m_data.blue = blueValue; }
inline Float Color::GetBlue() const { return m_data.blue; }
inline void Color::SetAlpha(Float alphaValue) { m_data.alpha = alphaValue; }
inline Float Color::GetAlpha() const { return m_data.alpha; }

inline const Color::Data& Color::GetData() const { return m_data; }
inline bool Color::operator !=(const Color& color) const { return !(*this == color); }

inline Float& Color::Data::operator [] (UInt8 index)
{
    if (index > 3) {
        FEATSTD_DEBUG_FAIL();
        index = 0;
    }

    return (&red)[index];
}

inline Float Color::Data::operator [] (UInt8 index) const
{
    if (index > 3) {
        FEATSTD_DEBUG_FAIL();
        return 0.0F;
    }

    CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(661, "Possible creation of out-of-bounds pointer");
    CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(662, "Possible creation of out-of-bounds pointer");
    return (&red)[index];
}

inline Float& Color::operator [] (UInt8 index)
{
    return m_data[index];
}

inline Float Color::operator [] (UInt8 index) const
{
    return m_data[index];
}

inline Color::operator Color::Data&() {
    return m_data;
}

inline Color::operator const Color::Data&() const {
    return m_data;
}

inline Color& Color::operator=(const Color::Data& color) {
    m_data = color;
    return *this;
}

/** @} */ // end of CommonBase
} // namespace Candera

#endif  // CANDERA_COLOR_H

