//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "Color.h"
#include <Candera/System/Mathematics/Math.h>

namespace Candera {
Color::Color()
{
    m_data.red   = 0.0F;
    m_data.green = 0.0F;
    m_data.blue  = 0.0F;
    m_data.alpha = 1.0F;
}

Color::~Color()
{
}

Color::Color(const Color &color) :
    m_data(color.m_data)
{
}

Color::Color(const Data &data) :
    m_data(data)
{
}

Color::Color(Float redValue, Float greenValue, Float blueValue, Float alphaValue)
{
    m_data.red = redValue;
    m_data.green = greenValue;
    m_data.blue = blueValue;
    m_data.alpha = alphaValue;
}

void Color::Set(Float redValue, Float greenValue, Float blueValue, Float alphaValue)
{
    m_data.red = redValue;
    m_data.green = greenValue;
    m_data.blue = blueValue;
    m_data.alpha = alphaValue;
}

void Color::SetHSV(Float hue, Float saturation, Float value, Float alphaValue)
{
    //Implemented according to wikipedia article http://de.wikipedia.org/wiki/HSV-Farbraum#Transformation_von_RGB_und_HSV

    m_data.alpha = alphaValue;

    hue = Math::RadianToDegree(hue);

    hue -= Math::Floor(hue / 360.0F) * 360.0F;

    if (hue == 360.0F) {
        hue = 0.0F;
    }

    if (saturation != 0.0F) {
        Float f = hue / 60.0F;
        Int32 hi = static_cast<Int32>(f);
        f = f - static_cast<Float>(hi);

        Float p = value * (1.0F - saturation);
        Float q = value * (1.0F - (saturation * f));
        Float t = value * (1.0F - (saturation * (1.0F - f)));

        switch (hi) {
            case 0:
                m_data.red = value;
                m_data.green = t;
                m_data.blue = p;
                break;
            case 1:
                m_data.red = q;
                m_data.green = value;
                m_data.blue = p;
                break;
            case 2:
                m_data.red = p;
                m_data.green = value;
                m_data.blue = t;
                break;
            case 3:
                m_data.red = p;
                m_data.green = q;
                m_data.blue = value;
                break;
            case 4:
                m_data.red = t;
                m_data.green = p;
                m_data.blue = value;
                break;
            case 5:
                m_data.red = value;
                m_data.green = p;
                m_data.blue = q;
                break;
            default:
                FEATSTD_DEBUG_FAIL();
                m_data.red = 0.0F;
                m_data.green = 0.0F;
                m_data.blue = 0.0F;
                break;
        }
    }
    else {
        m_data.red = value;
        m_data.green = value;
        m_data.blue = value;
    }

    Clamp();
}

void Color::GetHSV(Float& hue, Float& saturation, Float& value, Float& alphaValue) const
{
    //Implemented according to wikipedia article http://de.wikipedia.org/wiki/HSV-Farbraum#Transformation_von_RGB_und_HSV
    alphaValue = m_data.alpha;

    UInt32 maxIndex = 0U;
    UInt32 minIndex = 0U;

    Float max = 0.0F;
    Float min = 0.0F;

    //look for max component
    if ((m_data.red >= m_data.green) && (m_data.red >= m_data.blue)) {
        maxIndex = 0U;
        max = m_data.red;
    }
    else if (m_data.green >= m_data.blue) {
        maxIndex = 1U;
        max = m_data.green;
    }
    else {
        maxIndex = 2U;
        max = m_data.blue;
    }

    //look for min component
    if ((m_data.red <= m_data.green) && (m_data.red <= m_data.blue)) {
        minIndex = 0U;
        min = m_data.red;
    }
    else if (m_data.green <= m_data.blue) {
        minIndex = 1U;
        min = m_data.green;
    }
    else {
        minIndex = 2U;
        min = m_data.blue;
    }

    if (maxIndex != minIndex) {
        switch (maxIndex) {
            case 0: //maxIndex = red
                hue = 60.0F * ((m_data.green - m_data.blue) / (max - min));
                break;
            case 1:  //maxIndex = green
                hue = 60.0F * (2.0F + ((m_data.blue - m_data.red) / (max - min)));
                break;
            case 2:  //maxIndex = blue
                hue = 60.0F * (4.0F + ((m_data.red - m_data.green) / (max - min)));
                break;
            default:
                hue = 0.0F;
                break;
        }
    }
    else {
        hue = 0.0F;
    }

    if (hue < 0.0F) {
        hue = hue + 360.0F;
    }

    hue = Math::DegreeToRadian(hue);

    saturation = (max == Float(0)) ? Float(0) : ((max - min) / max);

    value = max;
}

Color Color::operator+(const Color& rhs) const {
    Color col(*this);
    col += rhs;
    return col;
}

Color Color::operator-(const Color& rhs) const {
    Color col(*this);
    col -= rhs;
    return col;
}

Color Color::operator*(const Color& rhs) const {
    Color col(*this);
    col *= rhs;
    return col;
}

Color& Color::operator=(const Color& color)
{
    if (&color != this) {
        m_data = color.m_data;
    }
    return *this;
}

Color& Color::operator+=(const Color& rhs) {
    m_data.red += rhs.m_data.red;
    m_data.green += rhs.m_data.green;
    m_data.blue += rhs.m_data.blue;
    m_data.alpha += rhs.m_data.alpha;
    return *this;
}

Color& Color::operator-=(const Color& rhs) {
    m_data.red -= rhs.m_data.red;
    m_data.green -= rhs.m_data.green;
    m_data.blue -= rhs.m_data.blue;
    m_data.alpha -= rhs.m_data.alpha;
    return *this;
}

Color& Color::operator*=(const Color& rhs) {
    m_data.red *= rhs.m_data.red;
    m_data.green *= rhs.m_data.green;
    m_data.blue *= rhs.m_data.blue;
    m_data.alpha *= rhs.m_data.alpha;
    return *this;
}

bool Color::operator ==(const Color& color) const
{
    return ((Math::FloatAlmostEqual(m_data.red, color.m_data.red)) &&
        (Math::FloatAlmostEqual(m_data.green, color.m_data.green)) &&
        (Math::FloatAlmostEqual(m_data.blue, color.m_data.blue)) &&
        (Math::FloatAlmostEqual(m_data.alpha, color.m_data.alpha)));
}

Color Color::operator+(const Float f) const {
    Color col(*this);
    col += f;
    return col;
}

Color Color::operator-(const Float f) const {
    Color col(*this);
    col -= f;
    return col;
}

Color Color::operator*(const Float f) const {
    Color col(*this);
    col *= f;
    return col;
}

Color& Color::operator=(const Float f) {
    m_data.red = f;
    m_data.green = f;
    m_data.blue = f;
    m_data.alpha = f;
    return *this;
}

Color& Color::operator+=(const Float f) {
    m_data.red += f;
    m_data.green += f;
    m_data.blue += f;
    m_data.alpha += f;
    return *this;
}

Color& Color::operator-=(const Float f) {
    m_data.red -= f;
    m_data.green -= f;
    m_data.blue -= f;
    m_data.alpha -= f;
    return *this;
}

Color& Color::operator*=(const Float f) {
    m_data.red *= f;
    m_data.green *= f;
    m_data.blue *= f;
    m_data.alpha *= f;
    return *this;
}

void Color::Clamp() {
    m_data.red = (m_data.red < Float(0)) ? Float(0) : ((m_data.red > Float(1)) ? Float(1) : m_data.red);
    m_data.green = (m_data.green < Float(0)) ? Float(0) : ((m_data.green > Float(1)) ? Float(1) : m_data.green);
    m_data.blue = (m_data.blue < Float(0)) ? Float(0) : ((m_data.blue > Float(1)) ? Float(1) : m_data.blue);
    m_data.alpha = (m_data.alpha < Float(0)) ? Float(0) : ((m_data.alpha > Float(1)) ? Float(1) : m_data.alpha);
}

void Color::Invert() {
    Clamp();
    m_data.red = 1.0F - m_data.red;
    m_data.green = 1.0F - m_data.green;
    m_data.blue = 1.0F - m_data.blue;
}

Color Color::GetAverageColor(const Color& a, const Color& b) {
    return ((a + b) * 0.5F);
}

Color Color::Lerp(const Color& a, const Color& b, Float weight){
    return (a + ((b - a) * weight));
}
} // namespace Candera

