//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_CAMERAGROUP_H)
#define CANDERA_CAMERAGROUP_H

#include <Candera/EngineBase/Common/BaseStringBufferAppenders.h>
#include <Candera/Environment.h>
#include <Candera/EngineBase/Common/CanderaObject.h>
#include <Candera/System/Container/Vector.h>

namespace Candera {

    /** @addtogroup CommonBase
     *  @{
     */
#if defined(CANDERA_3D_ENABLED)
    class Camera;
#endif
#if defined(CANDERA_2D_ENABLED)
    class Camera2D;
#endif

    /**
     * CameraGroup is a container that groups cameras from multiple scenes.
     *
     * CameraGroup has an arbitrary number of 2D and/or 3D cameras from different
     *  scenes and it is used by applications to easily access and change camera settings
     *  uniformly, e.g. in transitions.
     */
    class CameraGroup : public CanderaObject
    {
        public:
#if defined(CANDERA_3D_ENABLED)
            /**
             * Add a Camera3D object.
             * @param camera Camera3D object to add.
             */
            void AddCamera(Camera* camera);

            /**
             * Remove Camera3D object.
             */
            void RemoveCamera(Int index);

            /**
             * Remove all Camera3D objects.
             */
            void ClearCameraList();

            /**
             * Retrieve number of Camera3D objects.
             */
            SizeType GetCameraCount() const;

            /**
             * Get Camera3D object at certain position.
             * @param index Position of requested Camera3D object.
             * @return The requested Camera3D object.
             */
            Camera* GetCamera(Int index);
            const Camera* GetCamera(Int index) const;
#endif
#if defined(CANDERA_2D_ENABLED)
            /**
             * Add a Camera2D object.
             * @param camera Camera2D object to add.
             */
            void AddCamera2D(Camera2D* camera);

            /**
             * Remove Camera2D object.
             */
            void RemoveCamera2D(Int index);

            /**
             * Remove all Camera2D objects.
             */
            void ClearCamera2DList();

            /**
             * Retrieve number of Camera2D objects.
             */
            SizeType GetCamera2DCount() const;

            /**
             * Get Camera2D object at certain position.
             * @param index Position of requested Camera2D object.
             * @return The requested Camera2D object.
             */
            Camera2D* GetCamera2D(Int index);
            const Camera2D* GetCamera2D(Int index) const;
#endif
            /**
             * Enable/Disable rendering of containing cameras.
             * @param renderingEnabled Flag specifying if cameras rendering should
             *  be enabled or not.
             */
            void SetRenderingEnabled(bool renderingEnabled);

        private:
#ifdef CANDERA_3D_ENABLED
            Candera::Internal::Vector<Camera*> m_camera3DList;
#endif
#ifdef CANDERA_2D_ENABLED
            Candera::Internal::Vector<Camera2D*> m_camera2DList;
#endif
    };

     /** @} */ // end of CommonBase

} // namespace Candera

#endif // CANDERA_CAMERAGROUP_H
