//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################


#if !defined(Candera_BitmapTextRenderContext_h)
#define Candera_BitmapTextRenderContext_h

#include <Candera/EngineBase/Common/Color.h>
#include <Candera/EngineBase/Common/Bitmap.h>
#include <Candera/TextEngine/TextEngine.h>

namespace Candera {

class Bitmap;

/** @addtogroup CommonBase
 *  @{
 */

        struct RenderFn {
            void(*RenderPixelSpanFn)(Int32 x, Int32 y, Int32 length, UInt8 alpha, UInt32 preparedPenColor, class Bitmap& bitmap);
            UInt32 (*PrepareRenderColorFn)(const Color &color);
        };

        /**
          * @brief Implements a TextRenderContext for Bitmaps
          */
        class BitmapTextRenderContext : public TextRendering::TextRenderContext {
            typedef TextRendering::TextRenderContext Base;

            public:
                /**
                 * Constructor that initializes the default context states.
                 */
                BitmapTextRenderContext(const TextRendering::TextRenderContext* referenceContext = 0);

                /**
                 * Candera text rendering will use this clipping rectangle to avoid blitting
                 * of invisible (i.e. completely outside clipping area) glyphs.
                 * @return current clipping rectangle
                 * @see TextRenderContext::GetClipRect
                 */
                virtual const TextRendering::TextRect& GetClipRect() const override;

                /**
                 * Blit the given glyph bitmap to the specified position in the render target
                 * @param x left coordinate of the glyph bitmap
                 * @param y top coordinate of the glyph bitmap
                 * @param glyph the glyph bitmap to be blitted
                 * @see TextRenderContext::Blit
                 */
                virtual void Blit(Int16 x, Int16 y, const TextRendering::GlyphBitmap& glyph) override;

                /**
                 *  Retrieves the glyph cache access.
                 *  @return The current glyph cache access
                 *  @see TextRenderContext::GetGlyphCacheAccess
                 */
                virtual TextRendering::GlyphCacheAccess* GetGlyphCacheAccess() const { return (0 != m_referenceContext) ? m_referenceContext->GetGlyphCacheAccess() : 0; }

                /**
                 * Retrieves the bitmap to render text into.
                 * @return bitmap to be used for text rendering
                 */
                Bitmap::SharedPointer GetBitmap() const {
                    CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1763, Candera::TextRendering::BitmapTextRenderContext::GetBitmap, CANDERA_LINT_REASON_NONCONST)
                    return m_bitmap;
                }

                /**  define the target for text rendering
                 *  Text will be rendered into the specified bitmap. The bitmap
                 *  object must not be destroyed prior to destruction of the
                 *  BitmapTextRenderContext.
                 *  @param bitmap the bitmap to be used for rendering
                 *  @return true if bitmap is accepted, false if the bitmap has an unsupported color format
                 */
                 bool SetBitmap(const Bitmap::SharedPointer& bitmap);

                /**
                 * Retrieves the current pen color.
                 * @return current pen color
                 */
                const Color& GetPenColor() const { return m_penColor; }

                /** Define the color for text rendering
                 * @param color the new pen color
                 */
                void SetPenColor(const Color &color);

                /**
                 * Returns currently active clipping rectangle.
                 * @return current active clipping rectangle
                 */
                const TextRendering::TextRect& ClipRect() const { return m_clipRect; }

                /** Define clipping rectangle for text rendering
                 *  @param rect new clipping rectangle
                 */
                void SetClipRect(const TextRendering::TextRect &rect);

                /**
                 * reset clipping rectangle
                 */
                void ResetClipRect();

                /**
                 * Returns currently active rendering area.
                 * @return current active rendering area (intersection of bitmap bounds and clipping rectangle
                 */
                const TextRendering::TextRect& RenderRect() const { return m_renderRect; }

                /**
                 * Returns the pen color translated to the target bitmap pixel format.
                 * @return the pen color translated to the target bitmap pixel format
                 */
                UInt32 GetPreparedPenColor() const { return m_preparedPenColor; }

            protected:
                const TextRendering::TextSize& RenderArea() const { return m_renderArea; }
                void SetRenderArea(const TextRendering::TextSize& area);

                void SetRenderFn(const RenderFn &renderFn) { m_renderFn = renderFn; }

                UInt32 PrepareColor(const Color &color) const;

            private:
                UInt32 m_preparedPenColor;
                Color m_penColor;
                TextRendering::TextSize m_renderArea;
                TextRendering::TextRect m_renderRect;
                TextRendering::TextRect m_clipRect;
                RenderFn m_renderFn;
                Bitmap::SharedPointer m_bitmap;
                const TextRendering::TextRenderContext* m_referenceContext;
        };

/** @} */ // end of CommonBase

}// namespace Candera

#endif// Candera_BitmapTextRenderContext_h
