//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "BitmapTextRenderContext.h"

#include <Candera/EngineBase/Common/BitmapGlyphCacheAccess.h>
#include <Candera/EngineBase/Common/PixelBakery.h>
#include <Candera/System/Mathematics/Math.h>
#include <CanderaPlatform/OS/MemoryPlatform.h>
#include <CanderaPlatform/OS/StringPlatform.h>

namespace Candera {

// ----------------------------------------------------------------------------
template<typename PixelFormat> static void RenderPixelSpan(Int32 x, Int32 y, Int32 length, UInt8 alpha, UInt32 preparedPenColor, Bitmap& bitmap)
{
    CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1764, context, CANDERA_LINT_REASON_NONCONST)

    Bitmap::PixelsResource pixelsResource(bitmap.GetPixelsResourceHandle());
    CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(826, "Necessary to cast byte adressable buffer to use the data stored inside.")
    typename PixelFormat::BaseType* p = FeatStd::aligned_cast<typename PixelFormat::BaseType*>(pixelsResource.GetMutableData());

    typename PixelFormat::BaseType fg;
    PixelFormat::ToBaseType(preparedPenColor, fg);

    UInt32 w = bitmap.GetWidth();
    UInt32 h = bitmap.GetHeight();
    p += (((h - 1) - y) * w) + x;

    while (length > 0) {
        static_cast<void>(PixelFormat::Blend(*p, fg, alpha));
        ++p;
        --length;
    }
}

// ----------------------------------------------------------------------------
template<typename PixelFormat> static UInt32 PrepareRenderColor(const Color &color)
{
    UInt32 prepared;
    typename PixelFormat::BaseType encoded;

    static_cast<void>(PixelFormat::Compose(encoded, color));
    PixelFormat::ToUInt32(encoded, prepared);

    return prepared;
}

struct BitmapToPixelFormatMapping {
    Bitmap::PixelFormat format;
    RenderFn renderFn;
};

#define pxFormatType(pxFormat) pxFormat##PixelBakeryFormat
#define MappingEntry(format, pxFormat) \
    {    Bitmap::format, \
        { &RenderPixelSpan<pxFormatType(pxFormat)>, &PrepareRenderColor<pxFormatType(pxFormat)> } \
    }

const Int32 cMappingEntryCount = 6;

static const BitmapToPixelFormatMapping gMappingTable[cMappingEntryCount] = {
    MappingEntry(RgbUnsignedBytePixelFormat,      Rgb),
    MappingEntry(RgbUnsignedShort565PixelFormat,  Rgb565),

    MappingEntry(RgbaUnsignedBytePixelFormat,      Rgba),
    MappingEntry(RgbaUnsignedShort4444PixelFormat, Rgba4444),
    MappingEntry(RgbaUnsignedShort5551PixelFormat, Rgba5551),

    MappingEntry(AlphaUnsignedBytePixelFormat,      A8)
};


// ----------------------------------------------------------------------------
BitmapTextRenderContext::BitmapTextRenderContext(const TextRendering::TextRenderContext* referenceContext) :
    Base(),
    m_preparedPenColor(~UInt32(0)),
    m_penColor(1.0F, 1.0F, 1.0F, 1.0F),
    m_bitmap(0),
    m_referenceContext(referenceContext)
{
    m_renderFn.PrepareRenderColorFn = 0;
    m_renderFn.RenderPixelSpanFn = 0;
}

// ----------------------------------------------------------------------------
bool BitmapTextRenderContext::SetBitmap(const Bitmap::SharedPointer& bitmap)
{
    SetRenderArea(TextRendering::TextSize(0, 0));

    bool ok = true;
    m_bitmap = bitmap;
    if (bitmap != 0) {
        Int i = 0;
        while ((i < cMappingEntryCount) && ((bitmap->GetPixelFormat() != gMappingTable[i].format))) {
            ++i;
        }

        if (i < cMappingEntryCount) {
            SetRenderFn(gMappingTable[i].renderFn);
            SetRenderArea(TextRendering::TextSize(Int16(bitmap->GetWidth()), Int16(bitmap->GetHeight())));
            m_preparedPenColor = PrepareColor(m_penColor);
        }
        else {
            ok = false;
        }
    }

    return ok;
}

// ----------------------------------------------------------------------------
const TextRendering::TextRect& BitmapTextRenderContext::GetClipRect() const
{
    return m_renderRect;
}

// ----------------------------------------------------------------------------
void BitmapTextRenderContext::Blit(Int16 x, Int16 y, const TextRendering::GlyphBitmap& glyph)
{
    TextRendering::TextRect glyphRect;
    glyphRect.SetLeft(x);
    glyphRect.SetTop(y);
    glyphRect.SetWidth(static_cast<Int16>(glyph.width));
    glyphRect.SetHeight(static_cast<Int16>(glyph.height));

    TextRendering::TextRect clipped = Intersect(m_renderRect, glyphRect);
    if (!clipped.IsEmpty()) {
        const UInt8* pixels = glyph.pixels;
        FEATSTD_DEBUG_ASSERT(0 != pixels);
        const UInt8* p = pixels +
                            ((clipped.GetTop() - glyphRect.GetTop()) * static_cast<Int16>(glyph.width)) +
                            (clipped.GetLeft() - glyphRect.GetLeft());

        for (Int16 ly = clipped.GetTop(); ly <= clipped.GetBottom(); ++ly) {
            Int16 lx = clipped.GetLeft();
            const UInt8 *spanBegin = p;

            while (lx <= clipped.GetRight()) {
                const UInt8 *spanEnd = spanBegin;
                Int16 xBegin = lx;
                do {
                    ++lx;
                    ++spanEnd;
                } while ((lx <= clipped.GetRight()) && (*spanBegin == *spanEnd));

                m_renderFn.RenderPixelSpanFn(static_cast<Int32>(xBegin), static_cast<Int32>(ly), static_cast<Int32>(lx - xBegin),
                    *spanBegin, GetPreparedPenColor(), m_bitmap.GetSharedInstance());
                spanBegin = spanEnd;
            }
            p += glyph.pitch;
        }
    }
}

// ----------------------------------------------------------------------------
void BitmapTextRenderContext::SetPenColor(const Color &color)
{
    m_penColor = color;
    m_preparedPenColor = PrepareColor(color);
}

// ----------------------------------------------------------------------------
void BitmapTextRenderContext::SetClipRect(const TextRendering::TextRect &rect)
{
    TextRendering::TextRect surface(TextRendering::TextCoordinate(0, 0), m_renderArea);
    if (rect.IsEmpty()) {
        m_renderRect = surface;
    }
    else {
        m_renderRect = Intersect(surface, rect);
    }
}

// ----------------------------------------------------------------------------
void BitmapTextRenderContext::ResetClipRect()
{
    SetClipRect(TextRendering::TextRect());
}

// ----------------------------------------------------------------------------
void BitmapTextRenderContext::SetRenderArea(const TextRendering::TextSize& area)
{
    m_renderArea = area;
    ResetClipRect();
}

inline UInt32 BitmapTextRenderContext::PrepareColor(const Color &color) const {
    if (m_renderFn.PrepareRenderColorFn != 0) {
        return m_renderFn.PrepareRenderColorFn(color);
    }
    else {
        return ~static_cast<UInt32>(0);
    }
}

}    // namespace Candera

