//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_TREE_MATCH_BASE_H)
#define CANDERA_TREE_MATCH_BASE_H

#include <Candera/Environment.h>

namespace Candera {

/** @addtogroup CloningBase
 *  @{
 */

/**
 * @brief This class matches nodes from a tree to another one with the same
 * node pattern. Such trees are usually the result of cloning.
 */
template<typename Node>
class TreeMatchBase {
public:
    /**
     * Function that searches for the match. It matches a source tree to a
     * destination tree with the same structure as the source.
     * @param source The node from the source tree, for which the match is 
     *          requested.
     * @param sourceRoot The root node of the source tree.
     * @param destinationRoot The root node of the tree being matched with the
     *          source tree.
     * @return A pointer to the matching node in the destination tree. 0 if
     *          the match fails, either because the source is not part of the
     *          source tree, or because the destination tree does not have the
     *          same structure.
     */
    static Node* Find(
        const Node* source, 
        const Node* sourceRoot, 
        Node* destinationRoot);

private:
    FEATSTD_MAKE_CLASS_STATIC(TreeMatchBase);

    static Int GetDepth(const Node* source, const Node* root);
    static const Node* GetAncestor(const Node* source, Int depth);
    static Node* FindChild(
        const Node* source, 
        const Node* sourceRoot, 
        Node* destinationRoot);
};

/** @} */ // end of CloningBase

template <typename Node>
Node* TreeMatchBase<Node>::Find(
    const Node* source, 
    const Node* sourceRoot, 
    Node* destinationRoot)
{
    Int depth = GetDepth(source, sourceRoot);
    if(depth < 0) {
        return 0;
    }

    // Move sourceRoot and destinationRoot toward source.
    while(depth > 0) {
        --depth;
        const Node* newSourceRoot = GetAncestor(source, depth);
        if (newSourceRoot == 0) {
            return 0;
        }
        destinationRoot = FindChild(newSourceRoot, sourceRoot, destinationRoot);
        if (destinationRoot == 0) {
            return 0;
        }
        sourceRoot = newSourceRoot;
    }
    return destinationRoot;
}

template <typename Node>
Int TreeMatchBase<Node>::GetDepth(const Node* source, const Node* root)
{
    Int depth = 0;
    for(;;) {
        if (source == 0) {
            return -1;
        }
        if (source == root) {
            return depth;
        }
        source = source->GetParent();
        ++depth;
    }
}

template <typename Node>
const Node* TreeMatchBase<Node>::GetAncestor(const Node* source, Int depth)
{
    while(depth > 0) {
        if (source == 0) {
            return 0;
        }
        source = source->GetParent();
        --depth;
    }
    return source;
}

template <typename Node>
Node* TreeMatchBase<Node>::FindChild(
    const Node* source, 
    const Node* sourceRoot, 
    Node* destinationRoot)
{
    const Node* sourceChild = sourceRoot->GetFirstChild();
    Node* destinationChild = destinationRoot->GetFirstChild();
    for(;;) {
        if ((sourceChild == 0) || (destinationChild == 0)) {
            return 0;
        }
        if (sourceChild == source) {
            return destinationChild;
        }
        sourceChild = sourceChild->GetNextSibling();
        destinationChild = destinationChild->GetNextSibling();
    }
}


} // namespace Candera


#endif  // CANDERA_TREE_MATCH_H
