//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_TREE_CLONER_BASE_H)
#define CANDERA_TREE_CLONER_BASE_H

#include <Candera/Environment.h>
#include <Candera/EngineBase/Common/TreeTraverserBase.h>
#include <Candera/EngineBase/Cloning/TreeCloneStrategy.h>

namespace Candera {

/** @addtogroup CloningBase
 *  @{
 */

/**
 * @brief This class clones scene subtrees.
 *
 * By default, without a specific NodeCloneStartegy, this class provides a
 * clone tree where each Node is a shallow clone of the original, obtained by
 * calling Node::Clone.
 * The shallow clone does not contain pointers to non-shared objects, like 
 * Listeners and Strategies. The only exception to this rule is "const Char*".
 * Shared pointers are copied, meaning that both the original and the clone
 * point to the same object.
 *
 * To change the default behavior, a NodeCloneStrategy may be attached.
 */
template<typename Node>
class TreeClonerBase {
public:
    typedef TreeCloneStrategy<Node, Node> NodeCloneStrategy;

    /**
     * Constructor. Instances of this class are allowed to configure the
     * cloning strategy.
     */
    TreeClonerBase();
    /**
     * Destructor. The strategy is not destroyed. Only the week reference is
     * released.
     */
    ~TreeClonerBase();

    /**
     * Create a clone of the given node tree.
     * After each node is cloned via Node::Clone, the clone strategy, if
     * available is applied to each clone node.
     * The lifetime of the result is handled by the caller, via Node::Dispose.
     * @param root A reference of the source tree.
     * @return A clone tree, or 0 if cloning failed.
     */
    Node* CreateClone(const Node& root);

    /**
     * Retrieve the clone strategy.
     * @return Strategy set by SetNodeCloneStrategy, or 0 if no strategy was set.
     */
    const NodeCloneStrategy* GetNodeCloneStrategy() const { return m_nodeCloneStrategy; }
    NodeCloneStrategy* GetNodeCloneStrategy() { return m_nodeCloneStrategy; }

    /**
     * Set the strategy used when cloning nodes.
     * This is a generic strategy that should usually be aware of the types
     * of nodes available in the scene.
     * The strategy lifetime should be guaranteed by the caller, until either
     * another strategy is set, or TreeClonerBase is destroyed.
     * Setting 0 will disable application if special cloning strategies.
     * @param val Strategy set to the cloner.
     */
    void SetNodeCloneStrategy(NodeCloneStrategy* val) { m_nodeCloneStrategy = val; }
private:
    FEATSTD_MAKE_CLASS_UNCOPYABLE(TreeClonerBase);
    NodeCloneStrategy* m_nodeCloneStrategy;


    /**
     * Class used for pairing nodes from the source with nodes from the clone.
     * Compared with TreeMatch, this class also works on incomplete trees.
     * It holds state, and should be fed nodes in the order provided by 
     * TreeTraverser.
     */
    class Pairing
    {
    public:
        typedef typename NodeCloneStrategy::Pair Pair;

        /**
         * Initialize the Pairing to the root node.
         * @return false if already initialized.
         */
        bool Initialize(const Pair& current);
        /**
         * Find the parent pair of the given node and its clone. Clone may not be
         * linked yet to the tree. Searching proceeds for the last set node pair.
         * @return A pair between the parent node and its clone.
         */
        Pair FindPairedParent(const Node* node) const;
        /**
         * Find the clone matching to the given node. Searching proceeds from the 
         * last set node pair. The clone should already be available within the
         * tree.
         * @return A pair between the node and its clone.
         */
        Pair FindPairedNode(const Node* node) const;

        /**
         * Retrieve the root pair set within initialization.
         * @return A pair between the root and its clone, or 0 if not initialized.
         */
        const Pair& GetRootPair() const { return m_root; }

        /**
         * Set a given pair as the last traversed pair.
         * This pair is the basis of future searches
         */
        void SetLastPair(const Pair& last) { m_last = last; }
    private:
        Pair m_root;
        Pair m_last;
    };

    /**
     * @brief Class that traverses the original tree and creates as shallow clone 
     * of each node. 
     *
     * The cloned nodes are linked into a tree with the same structure a the
     * source tree.
     */
    class ShallowCloneTraverser : public TreeTraverserBase<const Node>
    {
    public:
        typedef TreeTraverserBase<const Node> Base;
        typedef typename Base::TraverserAction TraverserAction;

        /**
         * Retrieve the clone tree.
         * After calling this function the lifetime of the tree is the 
         * responsibility of the caller.
         * @param Pointer to the clone tree. 0 if the clone is not available.
         */
        Node* GetClone() const;
    protected:
        // Override function from ConstTreeTraverser.
        virtual TraverserAction ProcessNode(const Node& node) override;
    private:
        typedef typename Pairing::Pair Pair;
        Pairing m_pairing;
    };


    /**
     * @brief Class that traverses the original tree applies deep clone strategies.
     */
    class DeepCloneTraverser : public TreeTraverserBase<const Node>
    {
    public:
        typedef TreeTraverserBase<const Node> Base;
        typedef typename Base::TraverserAction TraverserAction;

        /**
         * Constructor. All references are week.
         * @param root The root of the source tree.
         * @param cloneRoot The root of the clone tree. This tree should have the
         *      same structure as the clone.
         * @param nodeCloneStrategy The clone strategy applied to each node.
         */
        DeepCloneTraverser(
            const Node* root, 
            Node* cloneRoot, 
            NodeCloneStrategy *nodeCloneStrategy);
        /**
         * Destructor. Week references are released.
         */
        ~DeepCloneTraverser();

        /**
         * Proceed with traversing the trees. This shadows the function from
         * ConstTreeTraverser.
         */
        void Traverse();
    protected:
        using Base::Traverse;

        // Override function from ConstTreeTraverser.
        virtual TraverserAction ProcessNode(const Node& node) override;
    private:
        typedef typename Pairing::Pair Pair;
        NodeCloneStrategy *m_nodeCloneStrategy;
        Pairing m_pairing;
    };

};

/** @} */ // end of CloningBase

template<typename Node>
TreeClonerBase<Node>::TreeClonerBase() :
    m_nodeCloneStrategy(0)
{
}
template<typename Node>
TreeClonerBase<Node>::~TreeClonerBase() 
{
    m_nodeCloneStrategy = 0;
}
template<typename Node>
Node* TreeClonerBase<Node>::CreateClone(const Node& root)
{
    ShallowCloneTraverser traverser;
    traverser.Traverse(root);
    Node* clone = traverser.GetClone();
    if (clone != 0) {
        DeepCloneTraverser strategyTraverser(&root, clone, m_nodeCloneStrategy);
        strategyTraverser.Traverse();
    }
    return clone;
}


template<typename Node>
bool TreeClonerBase<Node>::Pairing::Initialize(
    const Pair& current)
{
    // Set up root.
    if (m_root.source == 0) {
        m_root = current;
        m_last = current;
        return true;
    }
    return false;
}
template<typename Node>
typename TreeClonerBase<Node>::Pairing::Pair 
    TreeClonerBase<Node>::Pairing::FindPairedParent(
        const Node* node) const
{
    Pair parent = m_last;
    // Find clone parent.
    if (node->GetParent() == parent.source){
        // Traversing has followed a child.
        // Found.
    }
    else {
        // Traversing followed a sibling. Find the sibling.
        const Node* sibling = 0;
        do {
            sibling = parent.source->GetNextSibling();

            FEATSTD_DEBUG_ASSERT(parent.source != m_root.source);
            parent.source = parent.source->GetParent();
            FEATSTD_DEBUG_ASSERT(m_last.clone != 0);
            parent.clone = parent.clone->GetParent();
        } while (node != sibling);
    }

    // m_last now points to the parent of the current node.
    FEATSTD_DEBUG_ASSERT(parent.source == node->GetParent());
    return parent;
}
template<typename Node>
typename TreeClonerBase<Node>::Pairing::Pair 
    TreeClonerBase<Node>::Pairing::FindPairedNode(
        const Node* node) const
{
    if (node == m_root.source) {
        return m_root;
    }

    Pair parent = FindPairedParent(node);
    Pair current(parent.source->GetFirstChild(), parent.clone->GetFirstChild());
    while ((current.source != 0) && (current.clone != 0) && (current.source != node)) {
        current.source = current.source->GetNextSibling();
        current.clone = current.clone->GetNextSibling();
    }

    return current;
}


template<typename Node>
Node* TreeClonerBase<Node>::ShallowCloneTraverser::GetClone() const
{
    return m_pairing.GetRootPair().clone;
}
template<typename Node>
typename TreeClonerBase<Node>::ShallowCloneTraverser::TraverserAction 
    TreeClonerBase<Node>::ShallowCloneTraverser::ProcessNode(const Node& node)
{
    // Clone current.
    Node* clone = node.Clone();
    if (clone == 0) {
        Node* rootClone = m_pairing.GetRootPair().clone;
        if (rootClone != 0) {
            rootClone->Dispose();
        }
        return Base::StopTraversing;
    }

    Pair current(&node, clone);

    // Set up root.
    if (m_pairing.Initialize(current)) {
        return Base::ProceedTraversing;
    }

    Pair parent = m_pairing.FindPairedParent(current.source);

    // Link clone to parent.
    if (!parent.clone->AddChild(clone)){
        return Base::StopTraversing;
    }
    m_pairing.SetLastPair(current);

    return Base::ProceedTraversing;
}

template<typename Node>
TreeClonerBase<Node>::DeepCloneTraverser::DeepCloneTraverser(
    const Node* root, 
    Node* cloneRoot, 
    NodeCloneStrategy *nodeCloneStrategy) :
    m_nodeCloneStrategy(nodeCloneStrategy)
{
    m_pairing.Initialize(Pair(root, cloneRoot));
}
template<typename Node>
TreeClonerBase<Node>::DeepCloneTraverser::~DeepCloneTraverser()
{
    m_nodeCloneStrategy = 0;
}
template<typename Node>
void TreeClonerBase<Node>::DeepCloneTraverser::Traverse()
{
    if (m_nodeCloneStrategy != 0) {
        const Node* root = m_pairing.GetRootPair().source;
        if(root != 0) {
            Base::Traverse(*root);
        }
    }
}
template<typename Node>
typename TreeClonerBase<Node>::DeepCloneTraverser::TraverserAction 
    TreeClonerBase<Node>::DeepCloneTraverser::ProcessNode(const Node& node)
{
    Pair current = m_pairing.FindPairedNode(&node);

    if (m_nodeCloneStrategy != 0) {
        m_nodeCloneStrategy->Execute(current, m_pairing.GetRootPair());
    }
    m_pairing.SetLastPair(current);

    return Base::ProceedTraversing;
}


} // namespace Candera


#endif  // CANDERA_TREE_CLONER_BASE_H
