//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_TREE_CLONE_STRATEGY_H)
#define CANDERA_TREE_CLONE_STRATEGY_H

#include <Candera/Environment.h>

namespace Candera {

/** @addtogroup CloningBase
 *  @{
 */

/**
 * @brief This represents a pair of nodes, typically a source node and its 
 * clone.
 *
 * The semantics of the binding depend on usage.
 * @tparam Node Type of the nodes.
 */
template<typename Node>
struct TreeClonePair {    
    const Node* source; ///< Pointer to the source node.
    Node* clone; ///< Pointer to a clone of the source node.

    TreeClonePair() : source(0), clone(0) {}
    TreeClonePair(const Node* src, Node* cln) : source(src), clone(cln) {}
};
 
/**
 * @brief This is an interface for strategies used during cloning by TreeCloner.
 *
 * Such strategies define additional copying that might be necessary beyond
 * Node::Clone.
 * @tparam ChildNode Type of the currently cloned node.
 * @tparam RootNode Type of the root node.
 */
template<typename ChildNode, typename RootNode>
class TreeCloneStrategy {
public:
    typedef TreeClonePair<ChildNode> Pair;
    typedef TreeClonePair<RootNode> RootPair;

    virtual ~TreeCloneStrategy() {}

    /**
     * Execute the cloning strategy on the current pair of nodes.
     * The clone tree already contains a clone of each node in the source.
     * The set of nodes on which the strategy was already executed is undefined.
     * @param current The pair of nodes on which the strategy should be executed.
     * @param root The root of the tree being cloned, and its clone tree.
     */
    virtual void Execute(const Pair& current, const RootPair& root) = 0;

    /**
     * Clear the internal state. This may help avoid sharing resources
     * between cloning passes.
     */
    virtual void Clear() = 0;
};

/** @} */ // end of CloningBase

} // namespace Candera


#endif  // CANDERA_TREE_CLONE_STRATEGY_H
