//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_SHARED_OBJECT_CLONE_STRATEGY_H)
#define CANDERA_SHARED_OBJECT_CLONE_STRATEGY_H

#include <Candera/Environment.h>
#include <Candera/EngineBase/Cloning/TreeCloneStrategy.h>

namespace Candera {

/** @addtogroup CloningBase
 *  @{
 */


/**
 * @brief This is an interface for strategies that define how shared objects are
 * cloned.
 * @tparam Type The type of the shared object.
 * @tparam Node The type of the nodes to which these objects are linked.
 */
 template<typename Type, typename Node>
class SharedObjectCloneStrategy {
public:
    typedef typename Type::SharedPointer SharedPointer; ///< Type of a pointer to the shared object;

    virtual ~SharedObjectCloneStrategy() {}

    /**
     * @brief This is the return type of the Clone function.
     *
     * It contains the clone if available, and the information whether the
     * clone was created just now, or referenced from previous creation.
     */
     struct Result {
        /**
         * Pointer to the shared object.
         */
        SharedPointer sharedObject;
        /**
         * True if the shared object is the result of the operation that 
         * created this object.
         */
        bool isNew;

        /**
         * Constructor.
         */
        Result() : isNew(false) {}
    };

    /**
     * Creates a clone of the source object, or returns a reference to a 
     * previously created clone. The root of the trees is provided for the
     * situation where lookup within the tree is required.
     * @param source The object that needs to be cloned.
     * @param root A pair of the root of the source tree and its clone.
     * @return return A Result object for the current clone operation.
     */
    virtual Result Clone(const SharedPointer& source, 
        const TreeClonePair<Node>& root) = 0;

    /**
     * Clear the internal state. This may help avoid sharing resources
     * between cloning passes.
     */
    virtual void Clear() = 0;
};

/** @} */ // end of CloningBase

} // namespace Candera


#endif  // CANDERA_SHARED_OBJECT_CLONE_STRATEGY_H
