//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "SplineInterpolationStrategy.h"
#include <Candera/EngineBase/Animation/KeyframeSequence.h>
#include <Candera/System/Diagnostics/Log.h>
#include <Candera/System/MemoryManagement/MemoryManagement.h>

namespace Candera { namespace Animation {
    using namespace Diagnostics;

    FEATSTD_LOG_SET_REALM(LogRealm::CanderaAnimation);

    FEATSTD_RTTI_DEFINITION(SplineInterpolationStrategy, InterpolationStrategy)

    SplineInterpolationStrategy::SharedPointer SplineInterpolationStrategy::Create()
    {
        return SplineInterpolationStrategy::SharedPointer(FEATSTD_NEW(SplineInterpolationStrategy));
    }

    namespace {
        static Float DotProduct(const Float (&a)[4], const Float (&b)[4])
        {
            return (a[0] * b[0]) + (a[1] * b[1]) + (a[2] * b[2]) + (a[3] * b[3]);
        }
    }

    void SplineInterpolationStrategy::Interpolate(const KeyframeSequence* keyframeSequence, SequenceTimeType sequenceTime, Float* resultValue) const
    {
        // caller should guarantee this
        if (keyframeSequence->GetKeyframeCount() < 2) {
            FEATSTD_LOG_ERROR("Interpolation failed, too few keyframes.");
            for (Int32 i = 0; i < keyframeSequence->GetNumberOfComponents(); ++i) {
                resultValue[i] = 0.0F;
            }
            return;
        }

        Int32 indexFarLeft = 0;
        Int32 indexLeft = 0;
        Int32 indexRight = 1;
        Int32 indexFarRight = 1;
        const_cast<Animation::KeyframeSequence*>(keyframeSequence)->Find2FramesAroundSequenceTime(sequenceTime, indexLeft, indexRight);
        if (indexLeft > 0) {
            indexFarLeft = indexLeft - 1;
        }
        else {
            indexFarLeft = indexLeft;
        }

        if ((indexRight + 1) < keyframeSequence->GetKeyframeCount())
        {
            indexFarRight = indexRight + 1;
        }
        else {
            indexFarRight = indexRight;
        }

        const KeyframeSequence::Keyframe farLeftKeyframe = keyframeSequence->GetKeyframe(indexFarLeft);
        const KeyframeSequence::Keyframe leftKeyframe = keyframeSequence->GetKeyframe(indexLeft);
        const KeyframeSequence::Keyframe rightKeyframe = keyframeSequence->GetKeyframe(indexRight);
        const KeyframeSequence::Keyframe farRightKeyframe = keyframeSequence->GetKeyframe(indexFarRight);

        Float s;
        if (rightKeyframe.m_sequencetimeMs != leftKeyframe.m_sequencetimeMs) {
            s = static_cast<Float>(sequenceTime - leftKeyframe.m_sequencetimeMs) / static_cast<Float>(rightKeyframe.m_sequencetimeMs - leftKeyframe.m_sequencetimeMs);
        }
        else {
            s = (sequenceTime <= rightKeyframe.m_sequencetimeMs) ? 0.0F : 1.0F;
        }
        const Float s2 = s * s;
        const Float s3 = s * s * s;

        // S, H
        const Float sVector[4] = { s3, s2, s, 1.0F };
        static const Float hMatrixTransposed[4][4] = {
            {  2.0F, -3.0F, 0.0F, 1.0F },
            { -2.0F,  3.0F, 0.0F, 0.0F },
            {  1.0F, -2.0F, 1.0F, 0.0F },
            {  1.0F, -1.0F, 0.0F, 0.0F }
        };

        // S * H
        const Float sh[4] = {
            DotProduct(sVector, hMatrixTransposed[0]),
            DotProduct(sVector, hMatrixTransposed[1]),
            DotProduct(sVector, hMatrixTransposed[2]),
            DotProduct(sVector, hMatrixTransposed[3])
        };

        const SequenceTimeType farLeftSeqMs = farLeftKeyframe.m_sequencetimeMs;
        const SequenceTimeType leftSeqMs = leftKeyframe.m_sequencetimeMs;
        const SequenceTimeType rightSeqMs = rightKeyframe.m_sequencetimeMs;
        const SequenceTimeType farRightSeqMs = farRightKeyframe.m_sequencetimeMs;

        for (Int32 i = 0; i < keyframeSequence->GetNumberOfComponents(); ++i) {
            // tangents
            const Float tangent0 = ((indexFarLeft == indexLeft) || (rightSeqMs == farLeftSeqMs)) ?
                Float(0)
                :
            (((rightKeyframe.m_valuesBegin[i] - farLeftKeyframe.m_valuesBegin[i]) *
                static_cast<Float>(rightSeqMs - leftSeqMs))/static_cast<Float>(rightSeqMs - farLeftSeqMs));

            const Float tangent1 = ((indexRight == indexFarRight) || (farRightSeqMs == leftSeqMs)) ?
                Float(0)
                :
            (((farRightKeyframe.m_valuesBegin[i] - leftKeyframe.m_valuesBegin[i]) *
                static_cast<Float>(rightSeqMs - leftSeqMs))/static_cast<Float>(farRightSeqMs - leftSeqMs));

            // C
            const Float c[4] = { leftKeyframe.m_valuesBegin[i], rightKeyframe.m_valuesBegin[i], tangent0, tangent1 };

            // SH * C
            resultValue[i] = DotProduct(sh, c);
        }
    }

    } // namespace Animation 
} // namespace Candera
