//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "LinearInterpolationStrategy.h"
#include <Candera/EngineBase/Animation/KeyframeSequence.h>
#include <Candera/System/Diagnostics/Log.h>

namespace Candera { namespace Animation {
    using namespace Diagnostics;

    FEATSTD_LOG_SET_REALM(LogRealm::CanderaAnimation);

    FEATSTD_RTTI_DEFINITION(LinearInterpolationStrategy, InterpolationStrategy)

    LinearInterpolationStrategy::SharedPointer LinearInterpolationStrategy::Create()
    {
        return LinearInterpolationStrategy::SharedPointer(FEATSTD_NEW(LinearInterpolationStrategy));
    }

    void LinearInterpolationStrategy::Interpolate(const KeyframeSequence* keyframeSequence, SequenceTimeType sequenceTime, Float* resultValue) const
    {
        // caller should guarantee this
        if (keyframeSequence->GetKeyframeCount() < 2) {
            FEATSTD_LOG_ERROR("Interpolation failed, too few keyframes.");
            for (Int32 i = 0; i < keyframeSequence->GetNumberOfComponents(); ++i) {
                resultValue[i] = 0.0F;
            }
            return;
        }

        Int32 indexLeft = 0;
        Int32 indexRight = 1;
        const_cast<Animation::KeyframeSequence*>(keyframeSequence)->Find2FramesAroundSequenceTime(sequenceTime, indexLeft, indexRight);
        const KeyframeSequence::Keyframe leftKeyframe = keyframeSequence->GetKeyframe(indexLeft);
        const KeyframeSequence::Keyframe rightKeyframe = keyframeSequence->GetKeyframe(indexRight);

        // calculate weighted sum for each component
        Float rightWeight;
        if (rightKeyframe.m_sequencetimeMs != leftKeyframe.m_sequencetimeMs) {
            rightWeight = static_cast<Float>(sequenceTime - leftKeyframe.m_sequencetimeMs) / static_cast<Float>(rightKeyframe.m_sequencetimeMs - leftKeyframe.m_sequencetimeMs);
        }
        else {
            rightWeight = (sequenceTime <= rightKeyframe.m_sequencetimeMs) ? 0.0F : 1.0F;
        }
        const Float leftWeight = 1.0F - rightWeight;

        for (Int32 j = 0; j < keyframeSequence->GetNumberOfComponents(); ++j) {
            resultValue[j] =
                (leftWeight * leftKeyframe.m_valuesBegin[j]) +
                (rightWeight * rightKeyframe.m_valuesBegin[j]);
        }
    }

    } // namespace Animation
} // namespace Candera
