//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "EaseInterpolationStrategy.h"
#include <Candera/EngineBase/Animation/KeyframeSequence.h>
#include <Candera/EngineBase/Animation/AbstractEasingFunction.h>
#include <Candera/System/Diagnostics/Log.h>

namespace Candera { namespace Animation {
    using namespace Diagnostics;

    FEATSTD_LOG_SET_REALM(LogRealm::CanderaAnimation);

    FEATSTD_RTTI_DEFINITION(EaseInterpolationStrategy, InterpolationStrategy)

    EaseInterpolationStrategy::EaseInterpolationStrategy() :
        Base(),
        m_easeDirection(EaseIn)
    {
    }

    EaseInterpolationStrategy::~EaseInterpolationStrategy()
    {
    }

    EaseInterpolationStrategy::SharedPointer EaseInterpolationStrategy::Create()
    {
        return EaseInterpolationStrategy::SharedPointer(FEATSTD_NEW(EaseInterpolationStrategy));
    }

    void EaseInterpolationStrategy::SetEaseFunction(const AbstractEasingFunction::SharedPointer& easeFunction)
    {
        m_easeFunction = easeFunction;

        if (m_easeFunction == 0) {
            FEATSTD_LOG_INFO("No ease function passed to EaseInterpolationStrategy. The default linear function will be used.");
        }
    }

    void EaseInterpolationStrategy::Interpolate(const KeyframeSequence* keyframeSequence, SequenceTimeType sequenceTime, Float* resultValue) const
    {
        // caller should guarantee this
        if (keyframeSequence == 0) {
            FEATSTD_LOG_ERROR("Interpolation failed, KeyframeSequence is null.");
            return;
        }
        if (keyframeSequence->GetKeyframeCount() < 2) {
            for (Int32 i = 0; i < keyframeSequence->GetNumberOfComponents(); ++i) {
                resultValue[i] = 0.0F;
            }
            return;
        }

        Int32 indexLeft = 0;
        Int32 indexRight = 1;
        const_cast<Animation::KeyframeSequence*>(keyframeSequence)->Find2FramesAroundSequenceTime(sequenceTime, indexLeft, indexRight);
        const KeyframeSequence::Keyframe leftKeyframe = keyframeSequence->GetKeyframe(indexLeft);
        const KeyframeSequence::Keyframe rightKeyframe = keyframeSequence->GetKeyframe(indexRight);

        // calculate weighted sum for each component
        Float rightWeight;
        if (rightKeyframe.m_sequencetimeMs != leftKeyframe.m_sequencetimeMs) {
            rightWeight = static_cast<Float>(sequenceTime - leftKeyframe.m_sequencetimeMs) / static_cast<Float>(rightKeyframe.m_sequencetimeMs - leftKeyframe.m_sequencetimeMs);
        }
        else {
            rightWeight = (sequenceTime <= rightKeyframe.m_sequencetimeMs) ? 0.0F : 1.0F;
        }

        if (m_easeFunction != 0) {
            switch (m_easeDirection) {
                case EaseIn: {
                    rightWeight = m_easeFunction->Resolve(rightWeight);
                    break;
                }
                case EaseOut: {
                    rightWeight = 1.0F - m_easeFunction->Resolve(1.0F - rightWeight);
                    break;
                }
                case EaseInOut: {
                    if (rightWeight < 0.5F) {
                        rightWeight = m_easeFunction->Resolve(2.0F * rightWeight) * 0.5F;
                    }
                    else {
                        rightWeight = 1.0F - ((m_easeFunction->Resolve(2.0F * (1.0F - rightWeight))) * 0.5F);
                    }
                    break;
                }
                default:
                    break;
            }
        }

        Float leftWeight = 1.0F - rightWeight;

        for (Int32 j = 0; j < keyframeSequence->GetNumberOfComponents(); ++j) {
            resultValue[j] =
                ((leftWeight) * leftKeyframe.m_valuesBegin[j]) +
                ((rightWeight) * rightKeyframe.m_valuesBegin[j]);
        }
    }

    } // namespace Animation
} // namespace Candera
