//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_BounceFunction_H)
#define CANDERA_BounceFunction_H

#include <Candera/EngineBase/Animation/AbstractEasingFunction.h>

namespace Candera { namespace Animation {

/** @addtogroup AnimationBase
 *  @{
 */
  
    /** 
     *  @brief BounceEaseFunction implements an animation ease function that starts at 0.0F bouncing a configurable
     *   number of times before reaching final value 1.0F.
     */
    class BounceEaseFunction : public AbstractEasingFunction {
        FEATSTD_TYPEDEF_BASE(AbstractEasingFunction);

        public:
            enum { MaxBounceCount = 255 }; ///< Maximum number of bounces that can be configured.

            FEATSTD_TYPEDEF_SHARED_POINTER(BounceEaseFunction);

            FEATSTD_SHARED_POINTER_CREATE_DECLARATION();
            /**
             *  Constructor
             */
            BounceEaseFunction();
                        
            /**
             *  Destructor
             */
            virtual ~BounceEaseFunction() override;

            /**
             *  Function solve method.
             *  @param inValue  Function inValue argument.
             *  @return computed output of function for given argument.
             */
            virtual Float Resolve(Float inValue) const override;

            /**
             *  Set RestitutionCoefficient.
             *  The restitution coefficient represents the ratio of speeds after and before each
             *  bounce. For example a ball hitting the ground with velocity V, will bounce back 
             *  with a velocity V*RestitutionCoefficient.
             *  The restitution coefficient can have any value between 0.0F (no bounce after first fall) to 
             *  1.0F (each bounce will have the same amplitude).
             *  @param restitutionCoefficient Restitution coefficient.
             */
            void SetRestitutionCoefficient(Float restitutionCoefficient);

            /**
             *  Get RestitutionCoefficient.
             *  @return restitution coefficient.
             */
            Float GetRestitutionCoefficient() const { return m_restitutionCoefficient; }
                        
            /**
             *  Set BounceCount.
             *  The bounce count sets the number of bounces that will be mapped in the [0..1] input interval.
             *   For example:
             *   - bounce count = 0 -> f(0.0F) = 0.0F, f(1.0F) = 1.0F
             *   - bounce count = 1 -> f(0.0F) = f(0.67F) = 0.0F, f(0.33F) = f(1.0F) = 1.0F (for 1.0F restitution coefficient).
             *   @param bounceCount Bounce count.
             */
            void SetBounceCount(UInt8 bounceCount);

            /**
             *  Get BounceCount.
             *  @return bounce count.
             */
            UInt8 GetBounceCount() const { return m_bounceCount; }

            FEATSTD_RTTI_DECLARATION();

        private:
            UInt8 m_bounceCount;
            mutable UInt8 m_indexCache;
            Float m_restitutionCoefficient;
            Float m_bounceTimes[MaxBounceCount + 1];
    };
   
 /** @} */ // end of AnimationBase
    
    } // namespace Animation
} // namespace Candera

#endif
