//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_BezierInterpolationStrategy_H)
#define CANDERA_BezierInterpolationStrategy_H

#include <Candera/EngineBase/Animation/InterpolationStrategy.h>
#include <Candera/System/MemoryManagement/SharedPointer.h>

namespace Candera { namespace Animation {
/** @addtogroup AnimationBase
 *  @{
 */

    class KeyframeSequence;

    /**
     *  @brief InterpolationStrategy implementation.
     *
     *  This interpolation strategy produces transition between keyframe values using
     *  Bezier control points. Each interpolated value is calculated using a polynomial
     *  function based on two keyframes around the given sequence time and any number of
     *  Bezier control keyframes.
     *
     *  Limitation: For calculating an interpolated value, only control points that have
     *  keyframe sequence time greater then the left keyframe, and less or equal then the
     *  right keyframe are taken into consideration.
     */
    class BezierInterpolationStrategy: public InterpolationStrategy {
        FEATSTD_TYPEDEF_BASE(InterpolationStrategy);

        public:
            FEATSTD_TYPEDEF_SHARED_POINTER(BezierInterpolationStrategy);

            FEATSTD_RTTI_DECLARATION();

            /**
             *  Creates a BezierInterpolationStrategy object.
             *  @return Pointer to the created BezierInterpolationStrategy object.
             */
            FEATSTD_SHARED_POINTER_CREATE_DECLARATION();

            /**
             *  Destructs an BezierInterpolationStrategy object.
             */
            virtual ~BezierInterpolationStrategy() override {}

            /**
             *  Calculate the interpolated value of a KeyframeSequence at a given sequence time.
             *  @param keyframeSequence              The sequence of keyframes the interpolation strategy should be applied on.
             *  @param sequenceTime                  The time at which the interpolated value should be calculated.
             *  @param resultValue [out]             The resulted interpolated value.
             */
            virtual void Interpolate(
                const KeyframeSequence* keyframeSequence,
                SequenceTimeType sequenceTime,
                Float* resultValue) const override;

            /**
             *  Sets the control keyframe sequence.
             *  @param controlKeyframeSequence The control keyframe sequence that is set.
             */
            void SetControlKeyframeSequence(const MemoryManagement::SharedPointer<KeyframeSequence>& controlKeyframeSequence) { m_controlKeyframeSequence = controlKeyframeSequence; }

            /**
             *  Retrieves the control keyframe sequence.
             *  @return The conrol keyframe sequence.
             */
            MemoryManagement::SharedPointer<KeyframeSequence> GetControlKeyframeSequence() const { return m_controlKeyframeSequence; }

        protected:
            BezierInterpolationStrategy():m_controlKeyframeSequence(0) {}

        private:
            MemoryManagement::SharedPointer<KeyframeSequence> m_controlKeyframeSequence;
    };

 /** @} */ // end of AnimationBase

    } // namespace Animation
} // namespace Candera

#endif
