//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "BezierInterpolationStrategy.h"
#include <Candera/EngineBase/Animation/KeyframeSequence.h>
#include <Candera/System/Diagnostics/Log.h>
#include <Candera/System/Container/Vector.h>
#include <Candera/System/Mathematics/Math.h>

namespace Candera { namespace Animation {
    using namespace Diagnostics;
    using namespace Candera::Internal;

    FEATSTD_LOG_SET_REALM(LogRealm::CanderaAnimation);

    FEATSTD_RTTI_DEFINITION(BezierInterpolationStrategy, InterpolationStrategy)

    BezierInterpolationStrategy::SharedPointer BezierInterpolationStrategy::Create()
    {
        return BezierInterpolationStrategy::SharedPointer(FEATSTD_NEW(BezierInterpolationStrategy));
    }

    void BezierInterpolationStrategy::Interpolate(const KeyframeSequence* keyframeSequence, SequenceTimeType sequenceTime, Float* resultValue) const
    {
        // caller should guarantee this
        if (keyframeSequence->GetKeyframeCount() < 2) {
            FEATSTD_LOG_ERROR("Interpolate failed, too few keyframes.");
            for (Int32 i = 0; i < keyframeSequence->GetNumberOfComponents(); ++i) {
                resultValue[i] = 0.0F;
            }
            return;
        }

        Int32 indexLeft;
        Int32 indexRight;
        const_cast<Animation::KeyframeSequence*>(keyframeSequence)->Find2FramesAroundSequenceTime(sequenceTime, indexLeft, indexRight);
        const KeyframeSequence::Keyframe leftKeyframe = keyframeSequence->GetKeyframe(indexLeft);
        const KeyframeSequence::Keyframe rightKeyframe = keyframeSequence->GetKeyframe(indexRight);

        if (leftKeyframe.m_sequencetimeMs > sequenceTime) {
            MemoryPlatform::Copy(resultValue, leftKeyframe.m_valuesBegin, static_cast<SizeType>(keyframeSequence->GetNumberOfComponents()) * sizeof(Float));
            return;
        }

        if (rightKeyframe.m_sequencetimeMs < sequenceTime) {
            MemoryPlatform::Copy(resultValue, rightKeyframe.m_valuesBegin, static_cast<SizeType>(keyframeSequence->GetNumberOfComponents()) * sizeof(Float));
            return;
        }

        //build a list of adjacent keyframes and associated control points
        Candera::Internal::Vector<KeyframeSequence::Keyframe> keyframeList;
        static_cast<void>(keyframeList.Add(leftKeyframe));
        if (m_controlKeyframeSequence != 0) {
            for (Int32 i = 0; i < m_controlKeyframeSequence->GetKeyframeCount(); i++) {
                const KeyframeSequence::Keyframe controlKeyframe = m_controlKeyframeSequence->GetKeyframe(i);
                if ((controlKeyframe.m_sequencetimeMs > leftKeyframe.m_sequencetimeMs) && (controlKeyframe.m_sequencetimeMs <= rightKeyframe.m_sequencetimeMs)) {
                    static_cast<void>(keyframeList.Add(controlKeyframe));
                }
            }
        }
        static_cast<void>(keyframeList.Add(rightKeyframe));

        //prepare a Float array for different temporary calculations
        UInt32 count = static_cast<UInt32>(keyframeList.Size());
        Float* tempValues = FEATSTD_NEW_ARRAY(Float, static_cast<SizeType>((count) * 3U));
        if (tempValues == 0) {
            FEATSTD_LOG_ERROR("Interpolate failed, out of memory.");
            for (Int32 i = 0; i < keyframeSequence->GetNumberOfComponents(); ++i) {
                resultValue[i] = 0.0F;
            }
            return;
        }

        //find S between 0 and 1, for which applying the Bezier algorithm on keyframe sequence times array
        //will result the sequence time at which we want to calculate the interpolated value using

        for (UInt32 i = 0; i < count; i++) {
            tempValues[i] = static_cast<Float>(keyframeList[i].m_sequencetimeMs);
        }

        //Repeatedly apply De Casteljau's algorithm for a set of points, until the difference between
        //the calculated sequence time and given one is insignificant. At each step, the intermediate
        //points resulted in the previous step are used to reduce the searched value domain.
        //The error halves at each step.
        Float err = 1.0F; // difference between the calculated sequence time and given one
        Float left = 0.0F; // left approximation for S
        Float right = 2.0F; // right approximation for S
        while ((!Math::FloatAlmostEqual(err, 0.0F)) && (!Math::FloatAlmostEqual(left, right))) {
            // tempValues is composed of two sequences:
            // tempValues[0..count-1] contains computed values that will be used for next step if
            //resulted sequence time is greater than the given one
            // tempValues[count-1..2*count - 1] contains computed values that will be used for next step if
            //resulted sequence time is smaller than the given one
            // tempValues[count - 1] stores the resulted sequence time at each step
            if (tempValues[count - 1] >= static_cast<Float>(sequenceTime)) {
                right = (left + right)*0.5F;
            }
            else {
                MemoryPlatform::Copy(&tempValues[0], &tempValues[count-1], static_cast<SizeType>(count) * sizeof(Float));
                left = (left + right)*0.5F;
            }

            //De Casteljau's algorithm
            for (UInt32 i = 0; i < (count - 1); i++) {
                tempValues[(2 * (count - 1)) - i] = tempValues[count - 1];
                for (UInt32 j = count - 1; j > i; j--) {
                    tempValues[j] = (tempValues[j-1] + tempValues[j]) * 0.5F;
                }
            }

            err = tempValues[count - 1] - static_cast<Float>(sequenceTime);
        }

        Float S = (left + right) * 0.5F;
        Float oneMinusS = 1.0F - S;

        //here the actual Bezier formula is applied for the keyframe values

        //tempValues array is split into 3 series:
        // 1, S, S^2, ... S^(count-1)
        // (1-S)^(count-1), ... (1-S)^2, (1-S), 1
        // (count-1)C(0), (count-1)C(1), ... (count-1)C(count-1)
        Float* pS = tempValues;
        Float* pOneMinusS = pS + count;
        Float* nCk = pOneMinusS + count;

        pS[0] = 1.0F;
        pOneMinusS[count - 1] = 1.0F;
        nCk[0] = 1.0F;
        for (UInt32 i = 1; i < count; i++) {
            pS[i] = pS[i - 1] * S;
            pOneMinusS[(count - i) - 1] = pOneMinusS[count - i] * oneMinusS;
            nCk[i] = (nCk[i -1] * static_cast<Float>(count - i)) / static_cast<Float>(i);
        }

        //calculate Bezier coefficients
        for (UInt32 i = 0; i < count; i++) {
            tempValues[i] = pS[i] * pOneMinusS[i] * nCk[i];
        }

        //calculate final values
        for (Int i = 0; i < keyframeSequence->GetNumberOfComponents(); i ++) {
             resultValue[i] = 0.0F;
             for (UInt32 j = 0; j < count; j ++) {
                 resultValue[i] += tempValues[j] * keyframeList[j].m_valuesBegin[i];
             }
        }

        FEATSTD_DELETE_ARRAY(tempValues);
    }
    
    } // namespace Animation
} // namespace Candera 
