//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_ANIMATIONTIMEDISPATCHER_H)
#define CANDERA_ANIMATIONTIMEDISPATCHER_H

#include <Candera/Environment.h>
#include <Candera/EngineBase/Animation/AnimationPlayerBase.h>
#include <Candera/System/Container/Vector.h>
#include <Candera/System/EntityComponentSystem/Entity.h>
#include <FeatStd/MemoryManagement/SharedPointer.h>

namespace Candera { namespace Animation {
    /** @addtogroup AnimationBase
     *  @{
     */

    /**
     * @brief Dispatches world time (as received from an application) to a collection of AnmationPlayers.
     */
    class AnimationTimeDispatcher : public EntityComponentSystem::Entity {
        public:
            FEATSTD_TYPEDEF_SHARED_POINTER(AnimationTimeDispatcher);

            /**
             *  Creates a AnimationTimeDispatcher object.
             *  @return A pointer to the created AnimationTimeDispatcher.
             */
            FEATSTD_SHARED_POINTER_CREATE_DECLARATION();


            /**
             *  Destructor
             */
            ~AnimationTimeDispatcher(); // non-virtual; not intended for subclassing

            /**
             *  Set the world time in milliseconds, and dispatches it to all players.
             *  @param worldTimeInMilliseconds The world time in milliseconds.
             *  @param deltaTimeInMilliseconds The delta time in milliseconds.
             */
            void Update(TimeType worldTimeInMilliseconds, TimeType deltaTimeInMilliseconds);

            /**
             *  Set the world time in milliseconds.
             *  @param worldTimeMs The world time in milliseconds that is set.
             */
            void SetWorldTimeMs(WorldTimeType worldTimeMs);

            /**
             *  Retrieves the world time in milliseconds.
             *  @return The world time in milliseconds.
             */
            WorldTimeType GetWorldTimeMs() const {
                return m_worldTimeMs;
            }

            /**
             *  Dispatches the world time that has been set to all players.
             */
            void DispatchTime();

            /**
             *  Adds the player to the collection of players that are supplied with world time by this dispatcher.
             *  Note: An AnimationPlayer can only be added to one AnimationTimeDispatcher.
             *  @param player AnimationPlayer to be added.
             *  @return true if successful
             */
            bool AddPlayer(MemoryManagement::SharedPointer<AnimationPlayerBase> player);

            /**
             *  Removes the player from the list of players.
             *  The player is only marked for removal. The actual removal happens on the next dispatch event.
             *  @param player Player that will be removed from the dispatchers player list.
             *  @return True, if the player is in the list and haven't been removed yet.
             */
            bool RemovePlayer(MemoryManagement::SharedPointer<AnimationPlayerBase> player);

            /**
             * Retrieves the number of players associated with this dispatcher.
             *
             * @return The number of players associated with this dispatcher.
             */
            UInt32 GetPlayerCount() const;

            /**
             * Retrieves the player with the given index. Use GetAnimationPlayerCount() to
             * obtain the upper bound of valid player indices.
             *
             * @param index The index of the player.
             * @return The player at the specified index.
             */
            MemoryManagement::SharedPointer<AnimationPlayerBase> GetPlayer(UInt32 index);

        private:
            bool m_isPlayerMarkedAsRemoved;
            WorldTimeType m_worldTimeMs;
            Internal::Vector<MemoryManagement::SharedPointer<AnimationPlayerBase> > m_players;

            CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1704, Candera::Animation::AnimationTimeDispatcher::AnimationTimeDispatcher, CANDERA_LINT_REASON_INSTANCESOBTAINABLE)
            AnimationTimeDispatcher();
            AnimationTimeDispatcher(const AnimationTimeDispatcher& other);
            AnimationTimeDispatcher& operator=(const AnimationTimeDispatcher& other);

            void RemoveMarkedPlayers();

            FEATSTD_SHARED_POINTER_DECLARATION();
    };

    /** @} */ // end of AnimationBase
    } // namespace Animation
} // namespace Candera

#endif    // CANDERA_ANIMATIONTIMEDISPATCHER_H
