//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_AnimationPropertySetter_H)
#define CANDERA_AnimationPropertySetter_H

#include <Candera/Environment.h>
#include <Candera/System/Rtti/Rtti.h>
#include <Candera/System/MemoryManagement/CanderaHeap.h>
#include <Candera/System/MemoryManagement/SharedPointer.h>

namespace Candera { namespace Animation {
    /** @addtogroup AnimationBase
     *  @{
     */

    /**
     * @brief Base class for all animation property setters.
     *
     * Provides method to set the value of an animated property.
     */
    class AnimationPropertySetter {
    public:
        FEATSTD_TYPEDEF_SHARED_POINTER(AnimationPropertySetter);

        FEATSTD_RTTI_DECLARATION();

        /**
         *  Constructor
         */
        AnimationPropertySetter();

        /**
         *  Destructor
         */
        virtual ~AnimationPropertySetter();

        /**
         *  Set the value.
         *  @param value The value that is set.
         */
        virtual void Set(const Float* value) = 0;

    private:

        CANDERA_SHARED_POINTER_DECLARATION();
    };


    /**
     * @brief Base class for animation property setters which require the property name.
     *
     * Provides methods to set and get the name of the animated property.
     */
    class NamedAnimationPropertySetter: public AnimationPropertySetter {
        FEATSTD_TYPEDEF_BASE(AnimationPropertySetter);

    public:
        FEATSTD_RTTI_DECLARATION();

        NamedAnimationPropertySetter():
            m_propertyName(0) {}
        virtual ~NamedAnimationPropertySetter() { m_propertyName = 0; }

        /**
         *  Set the name for the animated property.
         *  @param propertyName The name that is set.
         */
        void SetPropertyName(const Char* propertyName) { m_propertyName = propertyName; }

        /**
         *  Retrieves the name for the animated property.
         *  @return The name for the animated property.
         */
        const Char* GetPropertyName() const { return m_propertyName; }

    private:
        const Char* m_propertyName;
    };

    template<typename T>
    class ObjectPropertySetter: public NamedAnimationPropertySetter {
        public:
            void SetAnimatedObject(T* object) { m_object = object; }
            T* GetAnimatedObject() { return m_object; }
        private:
            T* m_object;
    };

    template <typename T, bool DoCreate = true>
    struct Factory {
        static MemoryManagement::SharedPointer<T> Create() { return MemoryManagement::SharedPointer<T>(FEATSTD_NEW(T)); }
    };

    template<typename T>
    struct Factory<T, false> {
        static MemoryManagement::SharedPointer<T> Create() { return MemoryManagement::SharedPointer<T>(0); }
    };

    /** @} */ // end of AnimationBase

    } // namespace Animation
} // namespace Candera
#endif
