//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_AnimationPlayerBase_H)
#define CANDERA_AnimationPlayerBase_H

#include <Candera/EngineBase/Common/BaseStringBufferAppenders.h>
#include <Candera/EngineBase/Common/CanderaObject.h>
#include <Candera/EngineBase/Common/StringIdentifier.h>
#include <Candera/EngineBase/Animation/AnimationTimeType.h>
#include <Candera/System/Container/SingleLinkedList.h>
#include <Candera/System/MemoryManagement/SharedPointer.h>

namespace Candera { namespace Animation {
        class AnimationTimeDispatcher;
        class AnimationGroupPlayer;
        class AnimationPlayerListener;
  /** @addtogroup AnimationBase
 *  @{
 */

    /**
     *  @brief AnimationPlayerBase is a base class for all AnimationPlayers that use an AnimationTimeDispatcher
     *   to control animations.
     *
     *  The AnimationPlayerBase interface supports:
     *   - Start/Pause/Resume/Stop an animation.
     *   - Finish a playing animation in a given time.
     *   - Play animation one time, a number of predefined times or infinitely .
     *   - Play animation from one given sequence time and for a given duration.
     *   - Play animation at a given speed.
     *
     *  An AnimationPlayerBase cannot be used standalone, it must be added to exactly one AnimationTimeDispatcher.
     */
    class AnimationPlayerBase: public CanderaObject
    {
        FEATSTD_TYPEDEF_BASE(CanderaObject);

        public:
            FEATSTD_TYPEDEF_SHARED_POINTER(AnimationPlayerBase);

            /**
             *  Virtual destructor.
             */
            virtual ~AnimationPlayerBase() override;

            /**
             *  Receive current world time from the associated AnimationTimeDispatcher.
             *  Based on the received world time, the AnimationPlayer controls its animation.
             *
             *  @param worldTimeMs World time that will be used for updating the animation state in milliseconds.
             */
            virtual void ReceiveTime(WorldTimeType worldTimeMs) = 0;

            /**
             *  Start animation playback.
             *
             *  The animation will not be started if:
             *   - It is already playing in a group.
             *   - It is already playing and Stop fails.
             *
             *  After the method is successfully executed, the player will be enabled.
             *   The sequence time of the beginning of the animation is linked to the
             *   current world time as given by the time dispatcher.
             *
             *  Different AnimationPlayerBase implementations can extend/replace this functionality.
             *
             *  @return true if the animation was started successfully, false otherwise.
             *
             *  @see IsEnabled
             */
            virtual bool Start();

            /**
             *  Stop animation playback.
             *
             *  Resuming is not possible. The player will be disabled.
             *  Different AnimationPlayerBase implementations can extend/replace this functionality.
             *
             *  @return true if the animation was enabled before stopping, false otherwise.
             *
             *  @see IsEnabled
             */
            virtual bool Stop();

            /**
             *  Finish animation playback.
             *
             *  @param timeToFinishMs   Time in milliseconds in which the end of the animation is assured. If the specified time is longer
             *       than the remaining of the animation, then the animation will be ended normally.
             *  @return true if the animation will be finished within the specified time, false otherwise.
             */
            virtual bool Finish(WorldTimeType timeToFinishMs);

            /**
             *  Pause animation playback.
             *
             *  Resuming is possible using Resume. The player will be disabled.
             *  Different AnimationPlayerBase implementations can extend/replace this functionality.
             *
             *  @return true if the animation was paused successfully, false otherwise.
             *
             *  @see IsEnabled
             */
            virtual bool Pause();

            /**
             *  Resume animation playback.
             *
             *  Resuming is possible only after the playback is paused. The player will be enabled.
             *  Different AnimationPlayerBase implementations can extend/replace this functionality.
             *
             *  @return true if the animation was resumed successfully, false otherwise.
             *
             *  @see IsEnabled
             */
            virtual bool Resume();

            /**
             *  Set the speed factor of the animation.
             *
             *  The default value is 1 (1x), meaning that the animation SequenceTime will run 1:1 with the WorldTime given by the
             *   AnimationTimeDispatcher. A value of 2x will double the animation playback speed, while a value of -1x will keep
             *   the speed but change the direction in which SequenceTime flows.
             *  If the animation is played in a group, the actual play speed factor will be the composite factor between the set one and
             *   the one from the group. For example if the speed factor for the animation is 2x, but it is played in a group whose
             *   speed factor is 2x, the playing speed factor is 4x.
             *
             *  @param speedFactor The speed factor of the animation playback.
             */
            virtual void SetSpeedFactor(Float speedFactor);

            /**
             *  Retrieve the speed factor of the animation.
             *  The default value is 1 (1x), meaning that the animation SequenceTime will run 1:1 with the WorldTime given by the
             *   AnimationTimeDispatcher. A value of 2x will double the animation playback speed, while a value of -1x will keep
             *   the speed but change the direction in which SequenceTime flows.
             *  If the animation is played in a group, the actual play speed factor will be the composite factor between the set one and
             *   the one from the group. For example if the speed factor for the animation is 2x, but it is played in a group whose
             *   speed factor is 2x, the playing speed factor is 4x.
             *  @return The speed factor of the animation.
             */
            Float GetSpeedFactor() const { return m_speedFactor; }

            /**
             *  Set the number of times the animation is repeated.
             *  Repeat means that after the animation is ended, it will be played again.
             *  @param repeatCount   Number of repeat times. The default is 1, meaning the animation does not
             *       repeat. The value 0 has the meaning of infinite playback.
             */
            void SetRepeatCount(Int repeatCount) { m_repeatCount = repeatCount; }

            /**
             *   Retrieve the number of times the animation is repeated.
             *   Repeat means that after the animation is ended, it will be played again.
             *   @return The number of times the animation is repeated.
             */
            Int GetRepeatCount() const { return m_repeatCount; }

            /**
             *  Check if AnimationPlayer is enabled or not.
             *
             *  Enabled means that the AnimationPlayer is in playing state, which can be obtained by
             *   calling Start or Resume, and can be reset by Pause or Stop.
             *
             *  @return true if the animation is currently playing, false otherwise.
             */
            bool IsEnabled() const { return m_isEnabled && (!m_isPaused); }

            /**
             *  Check if AnimationPlayer is paused or not.
             *
             *  @return true if the animation is currently paused, false otherwise.
             */
            bool IsPaused() const { return m_isPaused; }

            /**
             *  Retrieves AnimationTimeDispatcher object.
             *  @return AnimationTimeDispatcher object set for this AnimationPlayerBase object, or null if not set.
             */
            const AnimationTimeDispatcher* GetTimeDispatcher() const { return m_timeDispatcher; }

            /**
             *  Declaration of callback function PastTheEndCallBackFunction. See function SetPastTheEndCallbackFunction for further details.
             */
            typedef void (*PastTheEndCallbackFunction)(Int32 completedIterationsCount, MemoryManagement::SharedPointer<AnimationPlayerBase> player, void* userData);

            /**
             *  Adds an AnimationPlayerListener object.
             *  @param listener The lifetime of the listener is managed by the calling code. AnimationPlayer does not take ownership.
             *  @return True if successful, false otherwise.
             */
            bool AddAnimationPlayerListener(AnimationPlayerListener* listener);

            /**
             *  Removes an AnimationPlayerListener object.
             *  @param listener The lifetime of the listener is managed by the calling code. AnimationPlayer does not take ownership.
             *  @return True if successful, false otherwise.
             */
            bool RemoveAnimationPlayerListener(AnimationPlayerListener* listener);

            /**
             * Set StringIdentifier of animation owner (composite).
             * For composite animations, the identifier must be the identifier of
             *  the CompositeGroup that owns the animation. For global animations, 
             *  the identifier must be 0.
             * 
             * @param identifier Owner's identifier.
             */
            void SetOwnerId(StringIdentifier* identifier) { m_stringIdentifier.SetOwner(identifier); }

            /*
             * Set a string id that is unique in the owner's context.
             *
             * @param id String id.
             * @param disposer String disposer.
             */
            void SetStringId(const Char* id, StringIdentifier::DisposerFn disposer) { m_stringIdentifier.SetId(id, disposer); }

            /**
             * Get StringIdentifier of this animation.
             * @return StringIdentifier.
             */
            const StringIdentifier* GetStringId() const { return &m_stringIdentifier; }

           FEATSTD_RTTI_DECLARATION();

        protected:
            MemoryManagement::SharedPointer<AnimationGroupPlayer> m_group;

            CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1704, Candera::Animation::AnimationPlayerBase::AnimationPlayerBase, CANDERA_LINT_REASON_INSTANCESOBTAINABLE)
            AnimationPlayerBase();

            bool StartInGroup(const AnimationPlayerBase::SharedPointer& animation);
            void SendNotifications(Int playCount);

            Float GetCompositeSpeedFactor() const;
            Int GetPastTheEndCount() const { return m_pastTheEndCount; }
            WorldTimeType GetPauseDuration() const { return m_pauseDuration; }
            bool IsFinishing() const { return m_isFinishing; }

            bool UpdateIterationsCount(Int additionalIteratonsCount);

            /**
            *  Called when Animation is ended.
            */
            void NotifyListenersOnPastEnd(Int completedIterationsCount);

            /**
            *  Called when the Animation is finished.
            */
            void NotifyListenersOnFinish();

            /**
            *  Called when the Animation is started.
            */
            void NotifyListenersOnStart();

            /**
            *  Called when the Animation is stopped.
            */
            void NotifyListenersOnStop();

            /**
            *  Called when the Animation is resumed.
            */
            void NotifyListenersOnResume();

            /**
            *  Called when the Animation is paused.
            */
            void NotifyListenersOnPause();

            /**
            *  Called when the Play direction was changed.
            */
            void NotifyListenersOnDirectionChanged(Candera::Int direction);

        private:
            friend class AnimationTimeDispatcher;

            const AnimationTimeDispatcher* m_timeDispatcher;

            bool m_isEnabled;
            bool m_isPaused;
            bool m_isFinishing;
            Float m_speedFactor;
            Int m_pastTheEndCount;
            Int m_repeatCount;
            Int m_notificationDepth;

            WorldTimeType m_pauseTime;
            WorldTimeType m_pauseDuration;

            void* m_callbackUserData;
            StringIdentifier m_stringIdentifier;

            typedef FeatStd::Internal::SingleLinkedList<AnimationPlayerListener*> AnimationPlayerListenerContainer;
            AnimationPlayerListenerContainer m_animationPlayerListeners;
            AnimationPlayerListenerContainer m_pendingAddAnimationPlayerListeners;

            AnimationPlayerBase(const AnimationPlayerBase&);
            AnimationPlayerBase& operator=(const AnimationPlayerBase&);

            void NotificationLoopBegin() { ++m_notificationDepth; }
            void NotificationLoopEnd();
            bool IsListenerRemovalAllowed() const { return m_notificationDepth == 0; }

            void SetTimeDispatcher(const AnimationTimeDispatcher* dispatcher) { m_timeDispatcher = dispatcher; }

            // Make this class manageable by MemoryManagement::SharedPointer
            FEATSTD_SHARED_POINTER_DECLARATION();
    };

    } // namespace Animation
} // namespace Candera

#endif    // CANDERA_AnimationPlayerBase_H
