//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_AnimationPlayer_H)
#define CANDERA_AnimationPlayer_H

#include <Candera/EngineBase/Common/BaseStringBufferAppenders.h>
#include <Candera/EngineBase/Animation/AnimationPlayerBase.h>
#include <Candera/System/MemoryManagement/Disposer.h>
#include <Candera/EngineBase/Animation/AnimationController.h>

namespace Candera { namespace Animation {
/** @addtogroup AnimationBase
 *  @{
 */

    /**
     *  @brief AnimationPlayer is an AnimationPlayerBase that animates AnimationBlendedProperties.
     *
     *  The AnimationPlayerBase implements the AnimationPlayerBase interface and extends it with support for:
     *   - Play animation normal or reverse.
     *   - Repeat an animation by replaying or by bouncing forward and backward.
     */
    class AnimationPlayer: public AnimationPlayerBase
    {
        FEATSTD_TYPEDEF_BASE(AnimationPlayerBase);

        public:
            /**
             *  The play direction (forward, reverse).
             */
            enum PlayDirection {
                Forward = 0,    ///< Forward
                Reverse = 1     ///< Reverse
            };

            /**
             *  The repeat mode (replay, bounce).
             *  replay - Restart the animation with each repeat.
             *  bounce - Reverse the animation direction with each repeat.
             */
            enum RepeatMode {
                Replay = 0,     ///< Restart the animation with each repeat.
                Bounce = 1      ///< Reverse the animation direction with each repeat.
            };

            FEATSTD_TYPEDEF_SHARED_POINTER(AnimationPlayer);

            /**
             *  Named constructor.
             *  @return MemoryManagement::SharedPointer to the created AnimationPlayer object.
             */
            FEATSTD_SHARED_POINTER_CREATE_DECLARATION();

            /**
             *  Virtual destructor.
             */
            virtual ~AnimationPlayer() override;

            /**
             *  Receive current world time from the associated AnimationTimeDispatcher.
             *  Based on the received world time, the AnimationPlayer updates animation properties
             *   through an AnimationController.
             *  @param worldTimeMs World time that will be used for updating the animation state in milliseconds.
             */
            virtual void ReceiveTime(WorldTimeType worldTimeMs) override;

            /**
             *  Start animation playback.
             *  @return true if the animation was started successfully, false otherwise.
             *  @see AnimationPlayerBase::Start
             */
            virtual bool Start() override;

            /**
             *  Finish animation playback.
             *  If the given time is shorter than the remaining time until the animation end, the speed factor at which the
             *   animation is played will be increased to fulfill the requirement.
             *  @param timeToFinishMs   Time in milliseconds in which the end of the animation is assured. If the specified time is longer
             *       than the remaining of the animation, then the animation will be ended normally. The minimum timeToFinishMs is 1ms.
             *  @return true if the animation will be finished within the specified time, false otherwise.
             */
            virtual bool Finish(WorldTimeType timeToFinishMs) override;

            /**
             *  Resume animation playback.
             *  The animation is resumed from the last pause sequenceTime.
             *  @return true if the animation was resumed successfully, false otherwise.
             *  @see AnimationPlayerBase::Resume
             */
            virtual bool Resume() override;

            /**
             *  Set corresponding AnimationController.
             *  @param controller   AnimationController that will be associated to this player.
             *  @see AnimationController.
             */
            void SetController(const AnimationController::SharedPointer& controller);

            /**
             *  Retrieves the corresponding AnimationController.
             *  @return The corresponding AnimationController.
             */
            const AnimationController::SharedPointer GetController() const { return m_controller; }

            /**
             *  Set the start point of the animation.
             *  @param sequenceStartTimeMs   The point in sequence time where the animation
             *       playback shall start. The default value is 0(ms).
             */
            void SetSequenceStartTimeMs(SequenceTimeType sequenceStartTimeMs);

            /**
             *  Retrieves the start point of the animation.
             *  @return the sequence start time in milliseconds.
             */
            SequenceTimeType GetSequenceStartTimeMs() const { return m_sequenceStartTimeMs; }

            /**
             *  Set the duration of the animation.
             *  When that duration has elapsed, playback will stop or act according to the repeat settings.
             *  @param sequenceDurationMs    The duration of the animation playback. The default value is 1000(ms).
             */
            void SetSequenceDurationMs(SequenceTimeType sequenceDurationMs);

            /**
             *  Retrieves the duration of the animation.
             *  When that duration has elapsed, playback will stop or act according to the repeat settings.
             *  @return The sequence duration in milliseconds.
             */
            SequenceTimeType GetSequenceDurationMs() const { return m_sequenceDurationMs; }

            /**
             *  Set repeat mode.
             *  @param repeatMode     Specify the repeat mode.
             *  @see RepeatMode.
             */
            void SetRepeatMode(RepeatMode repeatMode);

            /**
             *  Retrieves the repeat mode. @see RepeatMode
             *  @return The repeat mode.
             */
            RepeatMode GetRepeatMode() const { return static_cast<RepeatMode>(m_repeatMode); }

            /**
             *  Set play direction.
             *  @param direction     Specify the play direction.
             *  @see PlayDirection.
             */
            void SetDirection(PlayDirection direction);

            /**
             *  Retrieves the play direction. @see PlayDirection
             *  @return The play direction (forward, reverse).
             */
            PlayDirection GetDirection() const { return static_cast<PlayDirection>(m_playDirection); }

            /**
             *  Set the speed factor of the animation.
             *  @param speedFactor   The speed factor of the animation playback.
             *  @see AnimationPlayerBase::SetSpeedFactor.
             */
            virtual void SetSpeedFactor(Float speedFactor) override;

            FEATSTD_RTTI_DECLARATION();

        private:
            AnimationController::SharedPointer m_controller;

            SequenceTimeType m_sequenceStartTimeMs;
            SequenceTimeType m_sequenceDurationMs;

            // UInt for enum. According to MISRA 9-6-2 Bit Fields must be explicitly signed or unsigned. MISRA 9-6-3: Bit-fields shall not have enum type.
            UInt m_repeatMode : 8;
            // UInt for enum. According to MISRA 9-6-2 Bit Fields must be explicitly signed or unsigned. MISRA 9-6-3: Bit-fields shall not have enum type.
            UInt m_playDirection : 8;

            CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1704, Candera::Animation::AnimationPlayer::AnimationPlayer, CANDERA_LINT_REASON_INSTANCESOBTAINABLE)
            AnimationPlayer();
            AnimationPlayer(const AnimationPlayer&);
            AnimationPlayer& operator=(const AnimationPlayer&);

            bool InitController();
            bool InitController(WorldTimeType worldReferenceTime, SequenceTimeType sequenceReferenceTime, Float speedFactor);

            SequenceTimeType GetActualSequenceStartTimeMs() const;
            bool IsReversed() const;

            void NotifyListenersOnPlayerDirectionChanged(PlayDirection direction);
    };

    } // namespace Animation
} // namespace Candera

#endif    // CANDERA_AnimationPlayer_H
