//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "AnimationPlayer.h"
#include <Candera/EngineBase/Animation/AnimationTimeDispatcher.h>
#include <Candera/EngineBase/Animation/AnimationController.h>
#include <Candera/EngineBase/Animation/AnimationBlendedProperty.h>
#include <Candera/EngineBase/Animation/AnimationKeyframeSequence.h>
#include <Candera/EngineBase/Animation/AnimationPropertySetter.h>
#include <Candera/EngineBase/Animation/RelativePropertySetter.h>
#include <Candera/System/Mathematics/Math.h>
#include <Candera/System/Diagnostics/Log.h>

#define GetSign(value) ((value > 0.0F) ? 1 : -1)

namespace Candera { namespace Animation {
    using namespace Diagnostics;
    using namespace MemoryManagement;

    FEATSTD_LOG_SET_REALM(LogRealm::CanderaAnimation);

    FEATSTD_RTTI_DEFINITION(AnimationPlayer, AnimationPlayerBase)

AnimationPlayer::AnimationPlayer() :
    Base(),
    m_controller(0),
    m_sequenceStartTimeMs(0),
    m_sequenceDurationMs(1000),
    m_repeatMode(Replay),
    m_playDirection(Forward)
{
    // No instructions.
}

AnimationPlayer::SharedPointer AnimationPlayer::Create()
{
    return AnimationPlayer::SharedPointer(FEATSTD_NEW(AnimationPlayer)());
}

AnimationPlayer::~AnimationPlayer()
{
    CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1540,Candera::Animation::AnimationPlayer::m_controller, CANDERA_LINT_REASON_ASSOCIATION)
}

void AnimationPlayer::ReceiveTime(WorldTimeType worldTimeMs)
{
    if (GetTimeDispatcher() == 0) {
        FEATSTD_LOG_ERROR("ReceiveTime failed, null time dispatcher.");
        return;
    }
    if (m_controller == 0) {
        FEATSTD_LOG_ERROR("ReceiveTime failed, null controller.");
        return;
    }
    if (!IsEnabled()) {
        FEATSTD_LOG_ERROR("ReceiveTime failed, AnimationPlayer received time while being disabled.");
        return;
    }

    static SequenceTimeType sequenceTimeZero = static_cast<SequenceTimeType>(0);

    Int playCount;
    SequenceTimeType elapsedSequenceTime;
    if (m_sequenceDurationMs != sequenceTimeZero) {
        elapsedSequenceTime = Math::Absolute(m_controller->MapToSequenceTime(worldTimeMs) - GetActualSequenceStartTimeMs());
        playCount = static_cast<Int>(elapsedSequenceTime / m_sequenceDurationMs);
        elapsedSequenceTime -= playCount * m_sequenceDurationMs;
        if ((elapsedSequenceTime == sequenceTimeZero) && (playCount != 0)) {
            elapsedSequenceTime = m_sequenceDurationMs;
            --playCount;
        }
    }
    else {
        playCount = GetRepeatCount() - GetPastTheEndCount();
        playCount = Math::Minimum<Int>(1, playCount);
        elapsedSequenceTime = sequenceTimeZero;
    }

    if (UpdateIterationsCount(playCount)) {
        if (playCount > 0) {
            static_cast<void>(InitController(
                worldTimeMs - static_cast<WorldTimeType>(static_cast<Float>(elapsedSequenceTime) / Math::Absolute(GetCompositeSpeedFactor())),
                GetActualSequenceStartTimeMs(),
                (IsReversed() ? -1.0F : 1.0F) * GetCompositeSpeedFactor()));
        }
        m_controller->Animate(worldTimeMs);
    }
    else {
        SequenceTimeType sequenceFinishTimeMs = m_sequenceStartTimeMs;
        if (!IsReversed()) {
            sequenceFinishTimeMs += GetSign(GetCompositeSpeedFactor()) * m_sequenceDurationMs;
        }
        m_controller->AnimateSequenceTime(sequenceFinishTimeMs);
    }

    SendNotifications(playCount);
}

bool AnimationPlayer::Start()
{
    if (!AnimationPlayerBase::Start()) {
        return false;
    }

    if (m_controller == 0) {
        FEATSTD_LOG_ERROR("AnimationPlayer start failed, controller == 0.");
        static_cast<void>(AnimationPlayerBase::Stop());
        return false;
    }

    static_cast<void>(InitController(
        GetTimeDispatcher()->GetWorldTimeMs(),
        GetActualSequenceStartTimeMs(),
        ((GetDirection() == Reverse) ? -1.0F : 1.0F) * GetCompositeSpeedFactor()));

    for (UInt32 i = 0; i < m_controller->GetNumberOfProperties(); ++i) {
        AnimationBlendedProperty::SharedPointer abp = m_controller->GetProperty(i);
        if (abp != 0) {
            AnimationPropertySetter::SharedPointer aps = abp->GetAnimationPropertySetter();
            if(aps != 0) {
                Animation::RelativePropertySetter* relativePS = Dynamic_Cast<Animation::RelativePropertySetter*>(aps.GetPointerToSharedInstance());
                if (relativePS != 0) {
                    relativePS->Initialize();
                }
            }
        }
    }

    //Ensure that animated values of first key frame are set at the beginning of an animation.
    m_controller->Animate(GetTimeDispatcher()->GetWorldTimeMs());

    return true;
}

bool AnimationPlayer::Finish(WorldTimeType timeToFinishMs)
{
    if (!AnimationPlayerBase::Finish(timeToFinishMs)) {
        return false;
    }

    if (m_controller == 0) {
        FEATSTD_LOG_ERROR("AnimationPlayer finish failed, controller == 0.");
        return false;
    }

    static WorldTimeType minimumTimeToFinish = static_cast<WorldTimeType>(1);
    if (timeToFinishMs < minimumTimeToFinish) {
        timeToFinishMs = minimumTimeToFinish;
    }

    const WorldTimeType worldTimeMs = GetTimeDispatcher()->GetWorldTimeMs();
    const SequenceTimeType sequenceTimeMs = m_controller->MapToSequenceTime(worldTimeMs);
    const SequenceTimeType elapsedSequenceTime = Math::Absolute(sequenceTimeMs - GetActualSequenceStartTimeMs());
    if (m_sequenceDurationMs > elapsedSequenceTime) {
        WorldTimeType actualTimeToFinish = static_cast<WorldTimeType>(static_cast<Float>(m_sequenceDurationMs - elapsedSequenceTime) / GetCompositeSpeedFactor());
        if (actualTimeToFinish > timeToFinishMs) {
            static_cast<void>(InitController(
                worldTimeMs,
                sequenceTimeMs,
                ((m_controller->GetSpeedFactor() * static_cast<Float>(actualTimeToFinish)) / static_cast<Float>(timeToFinishMs))));
        }
    }

    return true;
}

bool AnimationPlayer::Resume()
{
    if (!AnimationPlayerBase::Resume()) {
        return false;
    }

    if (!InitController(
        GetTimeDispatcher()->GetWorldTimeMs(),
        m_controller->MapToSequenceTime(GetTimeDispatcher()->GetWorldTimeMs() - GetPauseDuration()),
        m_controller->GetSpeedFactor())) 
    {
        return false;
    }

    return true;
}

void AnimationPlayer::SetController(const AnimationController::SharedPointer& controller)
{
    m_controller = controller;
}

void AnimationPlayer::SetSequenceStartTimeMs(SequenceTimeType sequenceStartTimeMs)
{
    m_sequenceStartTimeMs = sequenceStartTimeMs;

    static_cast<void>(InitController());
}

void AnimationPlayer::SetSequenceDurationMs(SequenceTimeType sequenceDurationMs)
{
    m_sequenceDurationMs = Math::Maximum(static_cast<SequenceTimeType>(0), sequenceDurationMs);

    static_cast<void>(InitController());
}

void AnimationPlayer::SetRepeatMode(RepeatMode repeatMode)
{
    m_repeatMode = repeatMode;

    static_cast<void>(InitController());
}

void AnimationPlayer::SetDirection(PlayDirection direction)
{
    NotifyListenersOnDirectionChanged(direction);

    m_playDirection = direction;

    static_cast<void>(InitController());
}

void AnimationPlayer::SetSpeedFactor(Float speedFactor)
{
    AnimationPlayerBase::SetSpeedFactor(speedFactor);

    static_cast<void>(InitController());
}

bool AnimationPlayer::InitController()
{
    if ((GetTimeDispatcher() == 0) || (m_controller == 0) || IsFinishing()) {
        return false;
    }

    const WorldTimeType worldTimeMs = GetTimeDispatcher()->GetWorldTimeMs();

    return InitController(
        worldTimeMs,
        m_controller->MapToSequenceTime(worldTimeMs),
        (IsReversed() ? -1.0F : 1.0F) * GetCompositeSpeedFactor());
}

bool AnimationPlayer::InitController(WorldTimeType worldReferenceTime, SequenceTimeType sequenceReferenceTime, Float speedFactor)
{
    if (m_controller == 0) {
        return false;
    }

    m_controller->SetWorldReferenceTime(worldReferenceTime);
    m_controller->SetSequenceReferenceTime(sequenceReferenceTime);
    m_controller->SetSpeedFactor(speedFactor);

    return true;
}

SequenceTimeType AnimationPlayer::GetActualSequenceStartTimeMs() const
{
    SequenceTimeType sequenceStartTimeMs = m_sequenceStartTimeMs;
    if (IsReversed()) {
        sequenceStartTimeMs += GetSign(GetCompositeSpeedFactor()) * m_sequenceDurationMs;
    }

    return sequenceStartTimeMs;
}

bool AnimationPlayer::IsReversed() const
{
    return (((m_playDirection == Forward) && (m_repeatMode == Bounce) && ((GetPastTheEndCount() % 2) == 1)) ||
            ((m_playDirection == Reverse) && ((m_repeatMode != Bounce) || ((GetPastTheEndCount() % 2) == 0))));
}

void AnimationPlayer::NotifyListenersOnPlayerDirectionChanged(PlayDirection direction)
{
    if (direction != GetDirection()) {
        Base::NotifyListenersOnDirectionChanged(direction);
    }
}

    } // namespace Animation
} // namespace Candera
