//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "AnimationKeyframeSequence.h"
#include <Candera/System/Diagnostics/Log.h>

namespace Candera { namespace Animation {
    using namespace MemoryManagement;
    using namespace Diagnostics;

    FEATSTD_LOG_SET_REALM(LogRealm::CanderaAnimation);

    AnimationKeyframeSequence::AnimationKeyframeSequence() :
        Base(),
        m_isPeriodic(false)
    {
    }

    AnimationKeyframeSequence::SharedPointer AnimationKeyframeSequence::Create()
    {
        return AnimationKeyframeSequence::SharedPointer(FEATSTD_NEW(AnimationKeyframeSequence)());
    }

    AnimationKeyframeSequence::~AnimationKeyframeSequence()
    {
    }

    void AnimationKeyframeSequence::Interpolate(SequenceTimeType sequenceTimeMs, Float* resultValues) const
    {
        // border case: no frames
        if ((GetKeyframeCount() == 0) || (GetKeyframeSequenceTimes() == 0) || (GetKeyframeValues() == 0)) {
            FEATSTD_LOG_ERROR("Null result in case of error");
            for (Int32 i = 0; i < GetNumberOfComponents(); ++i) {
                resultValues[i] = 0.0F;
            }
            return;
        }

        // border case: one frame
        if (GetKeyframeCount() == 1) {
            const Keyframe firstKeyframe = GetKeyframe(0);
            MemoryPlatform::Copy(resultValues, firstKeyframe.m_valuesBegin, static_cast<SizeType>(GetNumberOfComponents()) * sizeof(Float));
            return;
        }

        // normal case: at least 2 frames
        // sequenceTimeMs outside of range defined by keyframes? then:
        //    periodic sequence: adjust time s.t. it is in range
        //    one-shot sequence: return first/last value of sequence
        const Keyframe firstKeyframe = GetKeyframe(0);
        const Keyframe lastKeyframe = GetKeyframe(static_cast<Int>(GetKeyframeCount()) - 1);
        if (m_isPeriodic) {
            const SequenceTimeType sequenceDurationMs = lastKeyframe.m_sequencetimeMs - firstKeyframe.m_sequencetimeMs;
            if (sequenceTimeMs < firstKeyframe.m_sequencetimeMs) {
                const SequenceTimeType distanceToLastMs = lastKeyframe.m_sequencetimeMs - sequenceTimeMs;
                const Int numberOfDurationsToAdd = static_cast<Int>(distanceToLastMs/sequenceDurationMs);
                sequenceTimeMs += numberOfDurationsToAdd * sequenceDurationMs;
            }

            if (sequenceTimeMs > lastKeyframe.m_sequencetimeMs) {
                const SequenceTimeType distanceToFirstMs = sequenceTimeMs- firstKeyframe.m_sequencetimeMs;
                const Int numberOfDurationsToSubtract = static_cast<Int>(distanceToFirstMs/sequenceDurationMs);
                sequenceTimeMs -= numberOfDurationsToSubtract * sequenceDurationMs;
            }
        }
        else {
            if (sequenceTimeMs < firstKeyframe.m_sequencetimeMs) {
                MemoryPlatform::Copy(resultValues, firstKeyframe.m_valuesBegin, static_cast<SizeType>(GetNumberOfComponents()) * sizeof(Float));
                return;
            }

            if (sequenceTimeMs > lastKeyframe.m_sequencetimeMs) {
                MemoryPlatform::Copy(resultValues, lastKeyframe.m_valuesBegin, static_cast<SizeType>(GetNumberOfComponents()) * sizeof(Float));
                return;
            }
        }

        if ((sequenceTimeMs < firstKeyframe.m_sequencetimeMs) && (sequenceTimeMs <= lastKeyframe.m_sequencetimeMs)) {
            FEATSTD_LOG_ERROR("Sequence time invalid.");
        }

        if (m_interpolationStrategy != 0) {
            m_interpolationStrategy->Interpolate(this, sequenceTimeMs, resultValues);
        }
    }

    /******************************************************************************
     *  SetInterpolationStrategy
     ******************************************************************************/
    void AnimationKeyframeSequence::SetInterpolationStrategy(const InterpolationStrategy::SharedPointer& interpolationStrategy)
    {
        m_interpolationStrategy = interpolationStrategy;
    }

    } // namespace Animation
} // namespace Candera
