//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_AnimationGroupPlayer_H)
#define CANDERA_AnimationGroupPlayer_H

#include <Candera/EngineBase/Common/BaseStringBufferAppenders.h>
#include <Candera/Environment.h>
#include <Candera/EngineBase/Animation/AnimationPlayerBase.h>
#include <Candera/System/Container/LinkedList.h>

namespace Candera { namespace Animation {

/** @addtogroup AnimationBase
 *  @{
 */

/**
 *  @brief AnimationGroupPlayer is an AnimationPlayerBase that controls a hierarchy
 *   of AnimationPlayerBase components.
 *
 *  The configuration of the AnimationGroupPlayer consists of attaching AnimationPlayerBase
 *   components. Each component is added to the group one by one along with additional settings:
 *    - a predecessor (another group component). If a component does not have a predecessor set,
 *    its start time depends only on the groups start time. Otherwise its start time depends
 *    on predecessors start/finish time.
 *    - a predecessor dependence type (With/After). A component with a predecessor can be
 *    configured to start with its predecessor or after it is finished.
 *    - a delay. If a component has a delay defined and its predecessor is started/finished, it
 *    will not be automatically started, but prepared for start (its start WorldTime is calculated).
 *
 *  When an AnimationGroupPlayer is started, each component gets started according to the
 *   previously configured order. Each component plays based on its own settings, the group
 *   influencing only the speed factor at which the component plays. The group is considered finished
 *   when all of its components are finished.
 *
 *  Restrictions:
 *   - Each component can have maximum one predecessor.
 *   - Delays cannot be negative.
 *   - Cycles are not detected by the AnimationGroupPlayer. The application that configures
 *      the AnimationGroupPlayer should assure no cycle is introduced.
 */
class AnimationGroupPlayer : public AnimationPlayerBase
{
    friend class AnimationPlayerBase;
    FEATSTD_TYPEDEF_BASE(AnimationPlayerBase);

    public:
        FEATSTD_TYPEDEF_SHARED_POINTER(AnimationGroupPlayer);

        enum StartType {
            WithPrevious = 0,   ///< Start with predecessor.
            AfterPrevious       ///< Start after predecessor has finished.
        };

        /**
         *  Creates an AnimationGroupPlayer object.
         *
         *  @return MemoryManagement::SharedPointer to the created AnimationGroupPlayer object.
         */
        FEATSTD_SHARED_POINTER_CREATE_DECLARATION();

        /**
         *  Destructs an AnimationGroupPlayer object.
         */
        virtual ~AnimationGroupPlayer() override;

        /**
         *  Receive current world time from the associated AnimationTimeDispatcher.
         *
         *  Based on the received world time, any ready to start component (their start time was previously
         *   calculated and it is less or equal with the received one) will be started. Their start will
         *   imply also its successors that depend on its StartTime will calculate their start time.
         *
         *  @param worldTimeMs World time that will be used for updating the animation state in milliseconds.
         */
        virtual void ReceiveTime(WorldTimeType worldTimeMs) override;

        /**
         *  Start animation playback.
         *
         *  When an AnimationGroupPayer is started, it will start all its AnimationPlayerBase components
         *   that do not have a predecessor set. With each component start, all its successors that depend
         *   on its StartTime will calculate their start time.
         *
         *  @return true if the animation was started successfully, false otherwise.
         *
         *  @see AnimationPlayerBase::Start
         */
        virtual bool Start() override;

        /**
         *  Stop animation playback.
         *
         *  When an AnimationGroupPlayer is stopped all its playing components are stopped.
         *
         *  @return true if the animation was stopped successfully, false otherwise.
         *
         *  @see AnimationPlayerBase::Stop.
         */
        virtual bool Stop() override;

        /**
         *  Finish animation playback.
         *
         *  To ensure that the animation group is finished in the given time frame, all unfinished (started or not) components will be
         *   finished in the given time.
         *
         *  @param timeToFinishMs   Time in milliseconds in which the end of the animation is assured. If the specified time is longer
         *       than the remaining of the animation, then the animation will be ended normally. The minimum timeToFinishMs is 1ms.
         *  @return true if the animation will be finished within the specified time, false otherwise.
         */
        virtual bool Finish(WorldTimeType timeToFinishMs) override;

        /**
         *  Pause animation playback.
         *
         *  When an AnimationGroupPlayer is paused all of the paused components will be paused.
         *
         *  @return true if the animation was paused successfully, false otherwise.
         *
         *  @see AnimationPlayerBase::Pause
         */
        virtual bool Pause() override;

        /**
         *  Resume animation playback.
         *
         *  When an AnimationGroupPlayer is resumed all of the paused components will be resumed.
         *
         *  @return true if the animation was resumed successfully, false otherwise.
         *
         *  @see AnimationPlayerBase::Resume
         */
        virtual bool Resume() override;

        /**
         *  Add an AnimationPlayerBase to the AnimationGroup hierarchy.
         *
         *  @param player The AnimationPlayerBase that should be added to the group.
         *  @param predecessor The predecessor of the given player. If the predecessor is 0, the player will be started by the group
         *   at group start time.
         *  @param startType Type of the predecessor:
         *   - WithPrevious: the player start time will depend on the start time of its predecessor. default
         *   - AfterPrevious: the player start time will depend on the finish time of its predecessor.
         *  @param delay An optional delay that can be added to the start time defined by the predecessor and predecessorType.
         */
        void AddPlayer(const AnimationPlayerBase::SharedPointer& player, const AnimationPlayerBase::SharedPointer& predecessor = AnimationPlayerBase::SharedPointer(0), StartType startType = WithPrevious, WorldTimeType delay = static_cast<WorldTimeType>(0));

        /**
         *  Remove an AnimationPlayerBase from the AnimationGroup hierarchy.
         *
         *  @param player The AnimationPlayerBase that should be removed from the group.
         *  @param removeSuccessors specifies if the successors of the removed player should be also removed or just set
         *   their predecessor to 0.
         */
        void RemovePlayer(const AnimationPlayerBase::SharedPointer& player, bool removeSuccessors = false);

        /**
         *  Remove all AnimationPlayerBase components.
         */
        FEATSTD_LINT_CLEANUP_FUNCTION(Candera::AnimationGroupPlayer::RemoveAllPlayers)
        void RemoveAllPlayers();

        /**
         *  Check if a player is attached to the group and retrieve its configuration if so.
         *
         *  If the animation is part of the group, the method will return the last configuration that was set
         *   through AddPlayer.
         *
         *  @param player The AnimationPlayerBase to be searched.
         *  @param predecessor The predecessor of the given player.
         *  @param startType Type of start relation (WithPrevioustart/AfterPrevious)
         *  @param delay Delay for the start time.
         *  @return true if player found in group, false otherwise.
         */
        bool FindPlayer(const AnimationPlayerBase::SharedPointer& player, AnimationPlayerBase::SharedPointer& predecessor, StartType& startType, WorldTimeType& delay);

        /**
         *  Get first successor for given player.
         *
         *  If the animation is part of the group and it has at least one successor of the given type, it will
         *   return the first successor. Next successors should be retrieved by calling GetNext with the result
         *   of this method.
         *
         *  Example of a method that iterates through all the components of a group:
         *
         *   void Iterate(SharedPointer<AnimationGroupPlayer> group, SharedPointer<AnimationPlayerBase> player) {
         *       if (player != 0) {
         *           Iterate(group, group->GetFirstSuccessor(player, AnimationGroupPlayer::WithPrevious));
         *           Iterate(group, group->GetFirstSuccessor(player, AnimationGroupPlayer::AfterPrevious));
         *           Iterate(group, group->GetNext(player));
         *       }
         *   }
         *
         *  The method should be called like:
         *
         *  Iterate(group, group->GetFirstSuccessor(SharedPointer<AnimationPlayer>(0), AnimationGroupPlayer::WithPrevious));
         *
         *  @param player The AnimationPlayerBase.
         *  @param startType Type of start relation (WithPrevioustart/AfterPrevious)
         *  @return first successor of the given type, if any. Otherwise 0.
         */
        const AnimationPlayerBase::SharedPointer GetFirstSuccessor(const AnimationPlayerBase::SharedPointer& player, StartType startType);

        /**
         *  Get next animation in list.
         *
         *  If the animation is part of the group the method will return the next animation that has
         *   the same predecessor and the same StartType as the given one.
         *
         *  @param player The AnimationPlayerBase.
         *  @return next animation in list if available, otherwise null.
         */
        const AnimationPlayerBase::SharedPointer GetNext(const AnimationPlayerBase::SharedPointer& player);

        /**
         *  Replace an AnimationPlayerBase inside an AnimationGroup.
         *
         *  @param player The AnimationPlayerBase to be replaced.
         *  @param newPlayer The new AnimationPlayerBase that will take over the previous player configuration.
         *  @return true if succeeded, false otherwise.
         */
        bool ReplacePlayer(const AnimationPlayerBase::SharedPointer& player, const AnimationPlayerBase::SharedPointer& newPlayer);

        FEATSTD_RTTI_DECLARATION();

    private:

        /**
         *  Commands that can be issued to a player component.
         */
        enum ComponentCommand {
            CNone,      ///< No command.
            CPrepare,   ///< Prepare the component to start by calculating its start time.
            CCheckTime, ///< Check if calculated start time is less than current time.
            CStart,     ///< Start the component.
            CPause,     ///< Pause the component.
            CResume,    ///< Resume paused component or update start time if it hasn't started yet.
            CStop,      ///< Stop component.
            CFinish,    ///< Finish the component in half of the remaining finish time.
            CEnd,       ///< Finish the component.
            CDispose    ///< Dispose the component.
        };

        class Component {
            public:

                /**
                 *  Different states of a component.
                 */
                enum ComponentState {
                    NotStarted,     ///< Has not started.
                    ReadyToStart,   ///< Is prepared to start.
                    Running,        ///< Is currently running.
                    Paused,         ///< Is currently paused.
                    Finished        ///< Has finished.
                };

                Component(AnimationPlayerBase::SharedPointer player) :
                    m_player(player),
                    m_predecessor(0),
                    m_successorsOnStart(0),
                    m_successorsOnFinish(0),
                    m_delay(static_cast<WorldTimeType>(0)),
                    m_startWorldTime(static_cast<WorldTimeType>(0)),
                    m_state(NotStarted)
                {
                }

                ~Component() {
                CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1540, Candera::AnimationGroupPlayer::Component::m_predecessor, CANDERA_LINT_REASON_ASSOCIATION)
                CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1540, Candera::AnimationGroupPlayer::Component::m_successorsOnStart, CANDERA_LINT_REASON_ASSOCIATION)
                CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1540, Candera::AnimationGroupPlayer::Component::m_successorsOnFinish, CANDERA_LINT_REASON_ASSOCIATION)
                };

                AnimationPlayerBase::SharedPointer m_player;
                Component* m_predecessor;
                Component* m_successorsOnStart;
                Component* m_successorsOnFinish;
                WorldTimeType m_delay;
                WorldTimeType m_startWorldTime;
                Internal::LinkedListNode<Component> m_listNode;
                ComponentState m_state;

                Component* Find(const AnimationPlayerBase::SharedPointer& player, bool removePlayer);
                void ProcessCommand(AnimationGroupPlayer* group, ComponentCommand command);
        };

        Int32 m_componentCount;
        Int32 m_finishedComponentCount;

        Component* m_firstComponent;
        WorldTimeType m_finishTime;


        Component* Find(const AnimationPlayerBase::SharedPointer& player, bool removePlayer = false);
        void ProcessCommand(Component* component, ComponentCommand command);

        void OnPlayerFinished(const AnimationPlayerBase::SharedPointer& player);

        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1704, Candera::AnimationGroupPlayer::AnimationGroupPlayer, CANDERA_LINT_REASON_INSTANCESOBTAINABLE)
        AnimationGroupPlayer();
        AnimationGroupPlayer(const AnimationGroupPlayer&);
        AnimationGroupPlayer& operator=(const AnimationGroupPlayer&);
};

    } // namespace Animation
} // namespace Candera

#endif    // CANDERA_AnimationGroupPlayer_H
