//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_AnimationController_H)
#define CANDERA_AnimationController_H

#include <Candera/Environment.h>
#include <Candera/EngineBase/Animation/AnimationTimeType.h>
#include <Candera/EngineBase/Animation/AnimationBlendedProperty.h>
#include <FeatStd/MemoryManagement/SharedPointer.h>
#include <FeatStd/Platform/CriticalSection.h>

namespace Candera { namespace Animation {

    /** @addtogroup AnimationBase
     *  @{
     */

    /**
     *  @brief AnimationController controls animations at a lower level than AnimationPlayer.
     *
     *  It provides methods to:
     *   - Animate associated properties at a given world time.
     *   - Animate associated properties at a given sequence time.
     *   - Map world time to sequence time.
     */
    class AnimationController {
        public:
            FEATSTD_TYPEDEF_SHARED_POINTER(AnimationController);

            /**
             *  Creates a AnimationController object.
             *  @return Pointer to the created AnimationController object.
             */
            FEATSTD_SHARED_POINTER_CREATE_DECLARATION();

            /**
             *  Destructor
             */
            ~AnimationController();// non-virtual; not intended for subclassing

            /**
             *  Update all properties for the given world time.
             *  @param worldTimeMs World time to calculate current values of animation properties.
             */
            void Animate(WorldTimeType worldTimeMs) const;

            /**
             *  Update all properties for the given sequence time.
             *  @param sequenceTimeMs Sequence time to calculate current values of animation properties.
             */
            void AnimateSequenceTime(SequenceTimeType sequenceTimeMs) const;

            /**
             *  Add a property to the animated properties list.
             *  @param prop Property to be animated.
             *  @return     True if property was added successfully. False otherwise, including prop = 0.
             */
            bool AddProperty(MemoryManagement::SharedPointer<AnimationBlendedProperty> prop);

            /**
             *  Retrieve number of animated properties.
             *  @return Number of attached animated properties.
             */
            SizeType GetNumberOfProperties() const;

            /**
             *  Retrieve a property from the animated properties list.
             *  @param index Index of the property.
             *  @return Animated property.
             */
            MemoryManagement::SharedPointer<AnimationBlendedProperty> GetProperty(SizeType index);

            /**
             *  Remove a property from the animated properties list.
             *  @param index Index of the property.
             *  @return True if property was removed successfully. False otherwise.
             */
            bool RemoveProperty(SizeType index);

            /**
             *  Set speed factor ( @see MapToSequenceTime ).
             *  @param speed Speed factor to set.
             */
            void SetSpeedFactor(Float speed) {
                m_speedFactor = speed;
            }

            /**
             *  Retrieves the speed factor ( @see MapToSequenceTime ).
             *  @return The speed factor.
             */
            Float GetSpeedFactor() const {
                return m_speedFactor;
            }

            /**
             *  Set world reference time in milliseconds ( @see MapToSequenceTime ).
             *  @param worldTimeMs World reference time to set, in ms.
             */
            void SetWorldReferenceTime(WorldTimeType worldTimeMs) {
                m_worldReferenceTime = worldTimeMs;
            }

            /**
             *  Retrieves the world reference time in milliseconds ( @see MapToSequenceTime ).
             *  @return The world reference time in milliseconds.
             */
            WorldTimeType GetWorldReferenceTime() const {
                return m_worldReferenceTime;
            }

            /**
             *  Set sequence reference time in milliseconds ( @see MapToSequenceTime ).
             *  @param sequenceTimeMs Sequence reference time to set, in ms.
             */
            void SetSequenceReferenceTime(SequenceTimeType sequenceTimeMs) {
                m_sequenceReferenceTime = sequenceTimeMs;
            }

            /**
             *  Retrieves the sequence reference time in milliseconds ( @see MapToSequenceTime ).
             *  @return The sequence reference time in milliseconds.
             */
            SequenceTimeType  GetSequenceReferenceTime() const {
                return m_sequenceReferenceTime;
            }

            /**
             *  Compute animation sequence time, based on the world time.
             *
             *  The basic formula is:
             *    sequenceTime = sequenceReferenceTime + speedFactor *  (worldTime - worldReferenceTime),
             *    where:
             *       - sequenceTime is the resulted sequence time.
             *       - sequenceReferenceTime is the sequence time from which the animation should start.
             *       - speedFactor defines the speed and direction the sequence time flows, relative to the world time.
             *       - worldTime is the current world time.
             *       - worldReferenceTime is the world time at which the animation started.
             *  @param worldTimeMs Current world time.
             *  @return Computed sequence time.
             */
            SequenceTimeType MapToSequenceTime(WorldTimeType worldTimeMs) const;

        private:
            typedef Internal::Vector<MemoryManagement::SharedPointer<AnimationBlendedProperty> > PropertyContainer;
            Float m_speedFactor;
            PropertyContainer m_properties;
            WorldTimeType m_worldReferenceTime;
            SequenceTimeType m_sequenceReferenceTime;

            FEATSTD_SHARED_POINTER_DECLARATION();

            CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1704, Candera::AnimationController::AnimationController, CANDERA_LINT_REASON_INSTANCESOBTAINABLE)
            AnimationController();
            AnimationController(const AnimationController& other);//lint !e1704 forbid until positive reason to allow
            AnimationController& operator=(const AnimationController& other);// forbid until positive reason to allow

#ifdef FEATSTD_THREADSAFETY_ENABLED
            mutable FeatStd::Internal::CriticalSection m_propertyContainerCriticalSection;
#endif
    };

    /** @} */ // end of AnimationBase
    }   // namespace Animation
} // namespace Candera
#endif
