//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "AnimationController.h"

#include <FeatStd/Platform/CriticalSectionLocker.h>

namespace Candera { namespace Animation {


AnimationController::AnimationController():
    m_speedFactor(1.0F),
    m_worldReferenceTime(0U),
    m_sequenceReferenceTime(0)
{
}

AnimationController::SharedPointer AnimationController::Create()
{
    return AnimationController::SharedPointer(FEATSTD_NEW(AnimationController)());
}


AnimationController::~AnimationController()
{
}

void AnimationController::Animate(WorldTimeType worldTimeMs) const
{
    const SequenceTimeType sequenceTimeMs = MapToSequenceTime(worldTimeMs);
    AnimateSequenceTime(sequenceTimeMs);
}

void AnimationController::AnimateSequenceTime(SequenceTimeType sequenceTimeMs) const
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    FeatStd::Internal::CriticalSectionLocker lock(&m_propertyContainerCriticalSection);
#endif
    const SizeType propertyCount = m_properties.Size();
    for (SizeType i = 0; i < propertyCount; ++i) {
        m_properties[i]->Animate(sequenceTimeMs);
    }
}

bool AnimationController::AddProperty(MemoryManagement::SharedPointer<AnimationBlendedProperty> prop)
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    FeatStd::Internal::CriticalSectionLocker lock(&m_propertyContainerCriticalSection);
#endif
    if (prop == 0) {
        return false;
    }
    return m_properties.Add(prop);
}

SizeType AnimationController::GetNumberOfProperties() const
{
    return m_properties.Size();
}

MemoryManagement::SharedPointer<AnimationBlendedProperty> AnimationController::GetProperty(SizeType index)
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    FeatStd::Internal::CriticalSectionLocker lock(&m_propertyContainerCriticalSection);
    if (index >= m_properties.Size()) {
        return AnimationBlendedProperty::SharedPointer(0);
    }
#endif
    return m_properties[index];
}

bool AnimationController::RemoveProperty(SizeType index)
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    FeatStd::Internal::CriticalSectionLocker lock(&m_propertyContainerCriticalSection);
#endif
    return m_properties.Remove(index);
}

SequenceTimeType AnimationController::MapToSequenceTime(WorldTimeType worldTimeMs) const
{
    return static_cast<SequenceTimeType>(static_cast<Float>(m_sequenceReferenceTime) + (m_speedFactor * static_cast<Float>(worldTimeMs - m_worldReferenceTime)));
}

    } // namespace Animation
} // namespace Candera
