//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Candera_ShaderParamSetter_h)
#define Candera_ShaderParamSetter_h

#include <Candera/Engine3D/ShaderParamSetters/AbstractShaderParamSetter.h>
#include <Candera/Engine3D/Core/Shader.h>
#include <Candera/Engine3D/Core/UniformNode.h>
#include <Candera/Engine3D/Core/ConstantAttributeNode.h>
#include <Candera/System/Container/LinkedList.h>
#include <Candera/System/Rtti/Rtti.h>

namespace Candera {
/** @addtogroup ShaderParamSetters3D
 *  @{
 */

//Forward declarations.
class Node;
class Appearance;

/**
 *  @brief ShaderParamSetter maintains a list of uniform parameters that are passed to a Shader.
 */
class ShaderParamSetter: public AbstractShaderParamSetter
{
    FEATSTD_TYPEDEF_BASE(AbstractShaderParamSetter);

    public:
        FEATSTD_TYPEDEF_SHARED_POINTER(ShaderParamSetter);
        /**
         *  Creates a ShaderParamSetter object.
         *  @return MemoryManagement::SharedPointer to the created ShaderParamSetter object.
         */
        static ShaderParamSetter::SharedPointer Create();

        /**
         *  Destructs this ShaderParamSetter object.
         */
        virtual ~ShaderParamSetter() override;

        /**
         *  Obtain a clone of this parameter setter.
         *  @return a clone of this parameter setter.
         */
        virtual AbstractShaderParamSetter::SharedPointer Clone() const override;

        /**
         *  Indicates whether this setter is suitable for draw call batching using instancing.
         *  @return  True if this setter supports shader parameter handling suitable for instancing, False otherwise.
         */
        virtual bool IsInstancingSupported() const { return true; }

        /**
         *  Creates a new uniform in the list if there is no uniform with the supplied name in the list.
         *  Sets the storage location if the uniform is already in the list.
         *  @param name specifies the uniform name referenced to in shader. The caller is responsible
         *  that the name string is alive until the ShaderParamSetter object is destructed. The ownership
         *  is not transfered and the caller is responsible for freeing the memory allocated for name.
         *  Note: Boolean and sampler uniform types require pointers to integer value data.
         *  @param name
         *  @param uniformType     Defines the data type of the uniform data given.
         *  @param data            Defines the pointer to the location where the data is stored.
         *  @param count           Specifies the number of array elements to load from the data given. For single elements this value is always 1.
         *  @param isDataVolatile  If true, the data pointer is not stored, but its content is copied instead.
         *  @return True on success. False, if the type and data parameters do not match, or if the list entry could not be created.
         */
        bool SetUniform(const Char* name, Shader::UniformType uniformType, Float* data, Int32 count = 1, bool isDataVolatile = false);

        /**
         *  Creates a new uniform in the list if there is no uniform with the supplied name in the list.
         *  Sets the storage location if the uniform is already in the list.
         *  @param name specifies the uniform name referenced to in shader. The caller is responsible
         *  that the name string is alive until the ShaderParamSetter object is destructed. The ownership
         *  is not transfered and the caller is responsible for freeing the memory allocated for name.
         *  Note: Boolean and sampler uniform types require pointers to integer value data.
         *  @param name
         *  @param uniformType     Defines the data type of the uniform data given.
         *  @param data            Defines the pointer to the location where the data is stored.
         *  @param count           Specifies the number of array elements to load from the data given. For single elements this value is always 1.
         *  @param isDataVolatile  If true, the data pointer is not stored, but its content is copied instead.
         *  @return True on success. False, if the type and data parameters do not match, or if the list entry could not be created.
         */
        bool SetUniform(const Char* name, Shader::UniformType uniformType, Int32* data, Int32 count = 1, bool isDataVolatile = false);

        /**
         *  Gets the type of the uniform name in output parameter type.
         *  @param name specifies the uniform name from which to retrieve the uniform type from.
         *  @param uniformType is an out-parameter that returns the type of the uniform data.
         *  @return true on success; false if there is no matching uniform in the list.
         */
        bool GetUniformType(const Char* name, Shader::UniformType& uniformType) const;

        /**
         *  Gets the uniform value storage in output parameter data.
         *  Note: Boolean and sampler uniform types are returned as integer data.
         *  @param name specifies the uniform name from which to retrieve the data location from.
         *  @param data is an out-parameter that returns the pointer to the uniform data.
         *  @return true on success; false if there is no matching uniform in the list, or the data parameter does not agree with the type
         *  of the found uniform.
         */
        bool GetUniformData(const Char* name, Float*& data) const;

        /**
         *  Gets the uniform value storage in output parameter data.
         *  Note: Boolean and sampler uniform types are returned as integer data.
         *  @param name specifies the uniform name from which to retrieve the data location from.
         *  @param data is an out-parameter that returns the pointer to the uniform data.
         *  @return true on success; false if there is no matching uniform in the list, or the data parameter does not agree with the type
         *  of the found uniform.
         */
        bool GetUniformData(const Char* name, Int32*& data) const;

        /**
         *  Gets the number of array elements to load from uniform data pointer. For single elements this value is 1.
         *  @param name specifies the uniform name to retrieve the count value from.
         *  @param count is an out-parameter that returns the number of elements of that uniform type.
         *  @return true on success; false if there is no matching uniform in the list.
         */
        bool GetUniformCount(const Char* name, Int32& count) const;

        /**
         *  Overrides AbstractShaderParamSetter::Activate.
         *  Activates all shader parameters set (see SetUniform) to the Shader.
         *  @param node Node      where the shader should be applied.
         *  @param appearance     Appearance pass on which the values should be set.
         *  @param instanceIndex  The index of the instance these parameters are applied to. A negative index applies parameters immediately.
         *  @return     True if shader parameters could be set, False otherwise. Also fails when appearance or appearance's
         *              shader is 0.
         */
        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1735, Candera::ShaderParamSetter::Activate, MISRA C++ Rule 8 - 3 - 1 allows the same default arguments for overriding functions)
        virtual bool Activate(Node& node, const MemoryManagement::SharedPointer<Appearance>& appearance, Int instanceIndex = -1) override;

        /**
         *  Get number of distinctive uniforms set with function SetUniform.
         *  @return Number of uniforms set by SetUniform.
         */
        SizeType GetNumberOfUniforms() const;

        /**
         *  Get name of the first uniform in the UniformList of this ShaderParamSetter object.
         *  @return Name of the first uniform, or 0 if no uniform is set with function SetUniform.
         */
        const Char* GetFirstUniformName();

        /**
         *  Get name of the next uniform in the UniformList of this ShaderParamSetter object.
         *  @return Name of the next uniform, or 0 if end of uniform list is reached.
         */
        const Char* GetNextUniformName();

        /**
         *  Check if the uniform type given is one of the Integer types.
         *  @param uniformType  Uniform type.
         *  @return True for:
         *   - Integer, IntegerVec2, IntegerVec3 or IntegerVec4
         *   - Bool, BoolVec2, BoolVec3 or BoolVec4
         *   - Sampler2D or SamplerCube
         */
        static bool IsIntegerType(Shader::UniformType uniformType);

        /**
         *  Check if the uniform type given is one of the Float types.
         *  @param uniformType  Uniform type.
         *  @return True for Float, FloatVec2, FloatVec3, FloatVec4, FloatMat2, FloatMat3 or FloatMat4.
         */
        static bool IsFloatType(Shader::UniformType uniformType);

        /**
         *  Creates a new constant attribute in the list if there is no constant attribute with the supplied semantic in the list.
         *  Alters it's value (data) if already present.
         *  Note: Constant vertex attributes outrule vertex arrays if they are present,
         *        if a constant vertex attribute is activated, it will be applied.
         *        This only works inside an OpenGL ES 2.0 or OpenGL 3.x Core or OpenGL 4.x Core context.
         *        Incompatibilities to OpenGL 2.0, OpenGL 3.x Compatibility profile or OpenGL 4.x Compatibility profile reside.
         *  @param semantic The semantic to set.
         *  @param attributeType defines the data type of the attribute data given.
         *  @param data defines the pointer to the location where the data is stored.
         *  @return true on success; false if the type and data parameters do not agree; or if the list entry could not be created.
         */
        bool SetConstantAttribute(ShaderParamNames::AttributeSemantic semantic, Shader::ConstantAttributeType attributeType, Float* data);

        /**
         *  Gets the type of the attribute semantic in output parameter type.
         *  @param semantic Specifies the attribute semantic from which to retrieve the attribute type from.
         *  @param attributeType is an out-parameter that returns the type of the uniform data.
         *  @return true on success; false if there is no matching attribute in the list.
         */
        bool GetConstantAttributeType(ShaderParamNames::AttributeSemantic semantic, Shader::ConstantAttributeType& attributeType) const;

        /**
         *  Gets the constant attribute value storage in output parameter data.
         *  @param semantic Specifies the attribute semantic from which to retrieve the data location from.
         *  @param data is an out-parameter that returns the pointer to the attribute data.
         *  @return true on success; false if there is no matching attribute in the list, or the data parameter does not agree with the type
         *  of the found attribute.
         */
        bool GetConstantAttributeData(ShaderParamNames::AttributeSemantic semantic, const Float*& data);

        /**
         *  Get number of distinctive constant attributes set with function SetConstantAttribute.
         *  @return Number of attributes set by SetConstantAttribute.
         */
        SizeType GetNumberOfConstantAttributes() const;

        /**
         *  Get semantic of the first constant attribute in the ConstantAttributeList of this ShaderParamSetter object.
         *  @return Semantic of the first attribute, or ShaderParamNames::AttributeSemanticCount if no constant attribute is set with function SetConstantAttribute.
         */
        ShaderParamNames::AttributeSemantic GetFirstConstantAttributeSemantic();

        /**
         *  Get semantic of the next constant attribute in the ConstantAttributeList of this ShaderParamSetter object.
         *  @return Semantic of the next constant attribute, or ShaderParamNames::AttributeSemanticCount if end of constant attribute list is reached.
         */
        ShaderParamNames::AttributeSemantic GetNextConstantAttributeSemantic();

        FEATSTD_RTTI_DECLARATION();

    protected:
        // Explicit protected Constructor and Copy-Constructor, use Create() to create an instance of this object
        ShaderParamSetter();
        ShaderParamSetter(const ShaderParamSetter& rhs);
        ShaderParamSetter& operator = (const ShaderParamSetter& vertexBuffer);

        /**
         *  Passes all uniforms set (see SetUniform) to the Shader, which is assumed to be active.
         *  @param shader         Shader to pass the set uniforms to, assumed to be active.
         *  @param instanceIndex  The index of the instance the uniforms are applied to. A negative index applies uniforms immediately.
         *  @return true if the operation succeeded for all uniforms.
         */
        bool ActivateUniformList(Shader& shader, Int instanceIndex = -1);

        /**
         *  Passes all constant attributes set (see SetConstantAttribute) to the Shader, which is assumed to be active.
         *  @param shader Shader to pass the set constant attributes to, assumed to be active.
         *  @return true if the operation succeeded for all constant attributes.
         */
        bool ActivateConstantAttributeList(Shader& shader);

    private:
        typedef Internal::LinkedList<Internal::UniformNode, &Internal::UniformNode::ListNode > UniformList;
        UniformList m_uniformList;              // Stores all uniforms.
        Internal::UniformNode* m_uniformNodeIterator;     // Used to iterate of the uniform list.

        typedef Internal::LinkedList<Internal::ConstantAttributeNode, &Internal::ConstantAttributeNode::ListNode> ConstantAttributeList;
        ConstantAttributeList m_constantAttributeList;
        Internal::ConstantAttributeNode* m_constantAttributeNodeIterator;

        // Finds the uniform with the supplied name in the list. 0 for not found.
        Internal::UniformNode* FindUniformNode(const Char* name) const { return const_cast<Internal::UniformNode*>(FindConstUniformNode(name)); }
        const Internal::UniformNode* FindConstUniformNode(const Char* name) const;

        bool CreateOrModifyUniformNode(const Char* name, Shader::UniformType uniformType, void* data, Int32 count, bool isDataVolatile);

        // Finds the uniform with the supplied name in the list. 0 for not found.
        Internal::ConstantAttributeNode* FindConstantAttributeNode(ShaderParamNames::AttributeSemantic semantic);
        const Internal::ConstantAttributeNode* FindConstConstantAttributeNode(ShaderParamNames::AttributeSemantic semantic) const;

        bool CreateOrModifyConstantAttributeNode(ShaderParamNames::AttributeSemantic semantic, Shader::ConstantAttributeType attributeType, void* data);
};
 /** @} */ // end of ShaderParamSetters3D
} // namespace Candera
#endif    // Candera_ShaderParamSetter_h
