//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "ShaderParamSetter.h"
#include <Candera/Engine3D/Core/Node.h>
#include <Candera/System/Diagnostics/Log.h>

namespace Candera {
    using namespace Internal;
    using namespace Diagnostics;

    FEATSTD_LOG_SET_REALM(LogRealm::CanderaEngine3D);

ShaderParamSetter::SharedPointer ShaderParamSetter::Create()
{
    CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(429, CANDERA_LINT_REASON_SHAREDPOINTER);
    ShaderParamSetter* ptr = FEATSTD_NEW(ShaderParamSetter)();
    if (ptr == 0) {
        FEATSTD_LOG_ERROR("Create ShaderParamSetter failed, out of memory.");
    }
    ShaderParamSetter::SharedPointer sharedPointer(ptr);
    return sharedPointer;
}

ShaderParamSetter::ShaderParamSetter() :
    m_uniformNodeIterator(0), 
    m_constantAttributeNodeIterator(0) 
{
}

ShaderParamSetter::ShaderParamSetter(const ShaderParamSetter& rhs) :
    Base(rhs),
    m_uniformNodeIterator(0), 
    m_constantAttributeNodeIterator(0) 
{
}

ShaderParamSetter::~ShaderParamSetter()
{
    if (m_uniformList.GetSize() > 0) {
        Shader::DeregisterUniformCacheAccessor(this);
    }

    CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1551, Candera::ShaderParamSetter::~ShaderParamSetter, none of these operations throw)
    UniformNode* u = &*m_uniformList.Begin();
    while (u != 0) {
        UniformNode* const next = u->ListNode.GetNext();
        FEATSTD_DELETE(u);
        u = next;
    }

    m_uniformNodeIterator = 0;

    ConstantAttributeNode* a = &*m_constantAttributeList.Begin();
    while (a != 0) {
        ConstantAttributeNode* const next = a->ListNode.GetNext();
        FEATSTD_DELETE(a);
        a = next;
    }

    m_constantAttributeNodeIterator = 0;
}

AbstractShaderParamSetter::SharedPointer ShaderParamSetter::Clone() const
{
    return ShaderParamSetter::SharedPointer(FEATSTD_NEW(ShaderParamSetter)(*this));
}

bool ShaderParamSetter::Activate(Node& node, const Appearance::SharedPointer& appearance, Int instanceIndex)
{
    FEATSTD_UNUSED(node); // Parameter unused when FEATSTD_LOG is disabled.
    if (appearance == 0) {
        FEATSTD_LOG_ERROR("Activation of ShaderParamSetter failed, appearance is null for Node:\"%s\".",
                          (node.GetName() == 0) ? "" : node.GetName());
        return false;
    }

    MemoryManagement::SharedPointer<Shader> shader = appearance->GetShader();
    if (shader == 0) {
        FEATSTD_LOG_ERROR("Activate ShaderParamSetter failed, shader is null.");
        return false;
    }

    // Set shader parameters.
    if (!ActivateUniformList(*shader, instanceIndex)) {
        FEATSTD_LOG_WARN("Activate ShaderParamSetter failed, extended shader parameter cannot be set.");
        return false;
    }

    if (!ActivateConstantAttributeList(*shader)) {
        FEATSTD_LOG_WARN("Activate ShaderParamSetter failed, constant attributes cannot be set.");
        return false;
    }

    return true;
}

bool ShaderParamSetter::SetUniform(const Char* name, Shader::UniformType uniformType, Float* data, Int32 count, bool isDataVolatile)
{
    if (!IsFloatType(uniformType)) {
        return false;
    }

    return CreateOrModifyUniformNode(name, uniformType, data, count, isDataVolatile);
}

bool ShaderParamSetter::SetUniform(const Char* name, Shader::UniformType uniformType, Int32* data, Int32 count, bool isDataVolatile)
{
    if (!IsIntegerType(uniformType)) {
        return false;
    }

    return CreateOrModifyUniformNode(name, uniformType, data, count, isDataVolatile);
}

bool ShaderParamSetter::CreateOrModifyUniformNode(const Char* name, Shader::UniformType uniformType, void* data, Int32 count, bool isDataVolatile)
{
    UniformNode* u = FindUniformNode(name);
    if (u == 0) {
        // new uniform
        UniformNode* newU = FEATSTD_NEW(UniformNode)(name, uniformType, data, count, isDataVolatile);
        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(774, new may return 0);
        if ((0 == newU) || (0 == newU->GetName())) {
            return false;
        }
        m_uniformList.Prepend(newU);
        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(429, newU is used/stored in m_uniformList);
        return true;
    }
    else {
        // Update existing uniform, type must match.
        if ((uniformType != u->GetType()) || (count != u->GetCount())) {
            return false;
        }

        u->SetData(data, isDataVolatile);
        return true;
    }
}

bool ShaderParamSetter::GetUniformType(const Char* name, Shader::UniformType& uniformType) const
{
    const UniformNode* const u = FindConstUniformNode(name);
    if (u == 0) {
        return false;
    }

    uniformType = u->GetType();
    return true;
}

bool ShaderParamSetter::GetUniformData(const Char *name, Float *&data) const
{
    UniformNode* u = FindUniformNode(name);
    if (u == 0) {
        data = 0;
        return false;
    }

    if (!IsFloatType(u->GetType())) {
        data = 0;
        return false;
    }

    data = FeatStd::Internal::PointerToPointer<Float*>(u->GetData());
    return true;
}

bool ShaderParamSetter::GetUniformData(const Char *name, Int32*& data) const
{
    UniformNode* u = FindUniformNode(name);
    if (u == 0) {
        data = 0;
        return false;
    }

    if (!IsIntegerType(u->GetType())) {
        data = 0;
        return false;
    }

    data = FeatStd::Internal::PointerToPointer<Int32*>(u->GetData());
    return true;
}

bool ShaderParamSetter::GetUniformCount(const Char* name, Int32& count) const
{
    const UniformNode* const uniformNode = FindConstUniformNode(name);
    if (uniformNode == 0) {
        return false;
    }

    count = uniformNode->GetCount();
    return true;
}

SizeType ShaderParamSetter::GetNumberOfUniforms() const
{
    return m_uniformList.GetSize();
}

const Char* ShaderParamSetter::GetFirstUniformName()
{
    if(GetNumberOfUniforms()!=0){
        m_uniformNodeIterator = &*m_uniformList.Begin();
        return m_uniformNodeIterator->GetName();
    }else{
        m_uniformNodeIterator=0;
        return 0;
    }
}

const Char* ShaderParamSetter::GetNextUniformName()
{
    if (m_uniformNodeIterator != 0) {
        m_uniformNodeIterator = m_uniformNodeIterator->ListNode.GetNext();
    }
    return (m_uniformNodeIterator != 0) ? m_uniformNodeIterator->GetName() : 0;
}

bool ShaderParamSetter::ActivateUniformList(Shader& shader, Int instanceIndex)
{
    if (0 == m_uniformList.GetSize()) {
        return true;
    }

    const void* accessorId = this;
    Shader::UniformCacheHandle uniformCacheHandle = shader.GetUniformCacheHandle(accessorId);
    bool isUniformCacheHandleValid = uniformCacheHandle.IsValid();

    if (isUniformCacheHandleValid && (shader.GetUniformCacheHandleLocationsCount(uniformCacheHandle) != m_uniformList.GetSize())) {
        // This case occurs when SetUniform was called after this setter has already been activated.
        bool wasSuccessful = shader.DeregisterUniformCacheAccessor(uniformCacheHandle);
        FEATSTD_DEBUG_ASSERT(wasSuccessful);
        FEATSTD_UNUSED(wasSuccessful);
        isUniformCacheHandleValid = false;
    }

    if (!isUniformCacheHandleValid) {
        if (!shader.RegisterUniformCacheAccessor(accessorId, &*(m_uniformList.Begin()), static_cast<UInt>(m_uniformList.GetSize()), uniformCacheHandle)) {
            FEATSTD_LOG_ERROR("ShaderParamSetter::ActivateUniformList failed, RegisterUniformCacheHandle failed.");
            return false;
        }
    }

    Int uniformLocation;
    UInt uniformCacheIndex = 0;
    bool isSuccessful = true;
    const UniformList::Iterator itEnd = m_uniformList.End();
    for (UniformList::Iterator it = m_uniformList.Begin(); it != itEnd; ++it) {
        const UniformNode* uniform = &*it;
        if (shader.GetUniformLocation(uniformCacheHandle, uniformCacheIndex++, uniformLocation)) {
            isSuccessful = isSuccessful && shader.SetUniform(uniform->GetData(), uniformLocation, uniform->GetType(), static_cast<UInt>(uniform->GetCount()), instanceIndex);
        }
        //else {
        //    FEATSTD_LOG_INFO("Uniform \"%s\" does not exist for current shader.", (uniformName == 0) ? "" : uniformName);
        //}
    }
    return isSuccessful;
}

const UniformNode* ShaderParamSetter::FindConstUniformNode(const Char *name) const
{
    const UniformList::ConstIterator itEnd = m_uniformList.End();
    for (UniformList::ConstIterator it = m_uniformList.Begin(); it != itEnd; ++it) {
        if (StringPlatform::CompareStrings(it->GetName(), name) == 0) {
            return &*it;
        }
    }
    return 0;
}

bool ShaderParamSetter::IsIntegerType(Shader::UniformType uniformType)
{
    return (uniformType == Shader::Integer) ||
        (uniformType == Shader::IntegerVec2) ||
        (uniformType == Shader::IntegerVec3) ||
        (uniformType == Shader::IntegerVec4) ||

        (uniformType == Shader::Bool) ||
        (uniformType == Shader::BoolVec2) ||
        (uniformType == Shader::BoolVec3) ||
        (uniformType == Shader::BoolVec4) ||

        (uniformType == Shader::Sampler2D) ||
        (uniformType == Shader::SamplerCube);
}

bool ShaderParamSetter::IsFloatType(Shader::UniformType uniformType)
{
    return (uniformType == Shader::Float) ||
        (uniformType == Shader::FloatVec2) ||
        (uniformType == Shader::FloatVec3) ||
        (uniformType == Shader::FloatVec4) ||

        (uniformType == Shader::FloatMat2) ||
        (uniformType == Shader::FloatMat3) ||
        (uniformType == Shader::FloatMat4);
}

bool ShaderParamSetter::SetConstantAttribute(ShaderParamNames::AttributeSemantic semantic, Shader::ConstantAttributeType attributeType, Float* data)
{
    return CreateOrModifyConstantAttributeNode(semantic, attributeType, data);
}

bool ShaderParamSetter::GetConstantAttributeType(ShaderParamNames::AttributeSemantic semantic, Shader::ConstantAttributeType& attributeType) const
{
    const ConstantAttributeNode* const can = FindConstConstantAttributeNode(semantic);
    if (can == 0) {
        return false;
    }

    attributeType = can->GetType();
    return true;
}

bool ShaderParamSetter::GetConstantAttributeData(ShaderParamNames::AttributeSemantic semantic, const Float*& data)
{
    const ConstantAttributeNode* can = FindConstantAttributeNode(semantic);
    if (can == 0) {
        data = 0;
        return false;
    }

    data = FeatStd::Internal::PointerToPointer<const Float*>(can->GetData());
    return true;
}

SizeType ShaderParamSetter::GetNumberOfConstantAttributes() const
{
    return m_constantAttributeList.GetSize();
}

ShaderParamNames::AttributeSemantic ShaderParamSetter::GetFirstConstantAttributeSemantic()
{
    if(m_constantAttributeList.GetSize()!=0){
        m_constantAttributeNodeIterator = &*m_constantAttributeList.Begin();
        return m_constantAttributeNodeIterator->GetSemantic();
    }else{
        m_constantAttributeNodeIterator = 0;
        return ShaderParamNames::AttributeSemanticCount;
    }
}

ShaderParamNames::AttributeSemantic ShaderParamSetter::GetNextConstantAttributeSemantic()
{
    if (m_constantAttributeNodeIterator != 0) {
        m_constantAttributeNodeIterator = m_constantAttributeNodeIterator->ListNode.GetNext();
    }
    return (m_constantAttributeNodeIterator != 0) ? m_constantAttributeNodeIterator->GetSemantic() : ShaderParamNames::AttributeSemanticCount;
}

bool ShaderParamSetter::ActivateConstantAttributeList(Shader& shader)
{
    Int attributeLocation;
    bool isSuccessful = true;
    const ConstantAttributeList::Iterator itEnd = m_constantAttributeList.End();
    for (ConstantAttributeList::Iterator it = m_constantAttributeList.Begin(); it != itEnd; ++it) {
        const ConstantAttributeNode* attributeNode = &*it;
        attributeLocation = shader.GetAttributeLocation(attributeNode->GetSemantic());
        if (attributeLocation != -1) {
            isSuccessful = isSuccessful && shader.SetConstantVertexAttribute(attributeNode->GetData(), attributeLocation, attributeNode->GetType());
        }
        else {
            FEATSTD_LOG_WARN("Constant attribute does not exist for current shader.");
        }
    }
    return isSuccessful;
}

ConstantAttributeNode* ShaderParamSetter::FindConstantAttributeNode(ShaderParamNames::AttributeSemantic semantic)
{
    const ConstantAttributeList::Iterator itEnd = m_constantAttributeList.End();
    for (ConstantAttributeList::Iterator it = m_constantAttributeList.Begin(); it != itEnd; ++it) {
        if (it->GetSemantic() == semantic) {
            return &(*it);
        }
    }
    return 0;
}

const ConstantAttributeNode* ShaderParamSetter::FindConstConstantAttributeNode(ShaderParamNames::AttributeSemantic semantic) const
{
    const ConstantAttributeList::ConstIterator itEnd = m_constantAttributeList.End();
    for (ConstantAttributeList::ConstIterator it = m_constantAttributeList.Begin(); it != itEnd; ++it) {
        if (it->GetSemantic() == semantic) {
            return &(*it);
        }
    }
    return 0;
}

bool ShaderParamSetter::CreateOrModifyConstantAttributeNode(ShaderParamNames::AttributeSemantic semantic, Shader::ConstantAttributeType attributeType, void* data)
{
    ConstantAttributeNode* can = FindConstantAttributeNode(semantic);
    if (can == 0) {
        // new uniform
        ConstantAttributeNode* newCan = FEATSTD_NEW(ConstantAttributeNode)(semantic, attributeType, data);
        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(774, new may return 0);
        if (newCan == 0) {
            return false;
        }
        m_constantAttributeList.Prepend(newCan);
        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(429, newCan is used/stored in m_constantAttributeList);
        return true;
    }
    else {
        can->SetData(data);
        return true;
    }
}

FEATSTD_RTTI_DEFINITION(ShaderParamSetter, AbstractShaderParamSetter)
} // namespace Candera
