//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Candera_GenericShaderParamSetter_h)
#define Candera_GenericShaderParamSetter_h

#include <Candera/Engine3D/ShaderParamSetters/ShaderParamSetter.h>
#include <Candera/Engine3D/Core/Light.h>
#include <Candera/System/MemoryManagement/MemoryManagement.h>
#include <Candera/System/Rtti/Rtti.h>
#include <Candera/System/Container/Vector.h>
#include <Candera/Environment.h>
#include <Candera/Macros.h>

namespace Candera {

/** @addtogroup ShaderParamSetters3D
 *  @{
 */

/**
* @brief GenericShaderParamSetter bundles uniform shader parameters that are calculated by Candera.
*        Each of those parameters can be enabled or disabled. If a parameter is enabled then it
*        will be calculated and passed to the shader,
*        if disabled the parameter is neither calculated nor passed.
*        Often needed shader params, namely ModelViewProjectionMatrix4, parameters for light, material and textures
*        are enabled by default. If a shader doesn't need them a possible optimization would be to disable these
*        parameters.
*/
class GenericShaderParamSetter: public ShaderParamSetter
{
    FEATSTD_TYPEDEF_BASE(ShaderParamSetter);

    public:
        FEATSTD_TYPEDEF_SHARED_POINTER(GenericShaderParamSetter);

        /**
        * Creates a GenericShaderParamSetter object.
        * @return MemoryManagement::SharedPointer to the created GenericShaderParamSetter.
        */
        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1511, Candera::GenericShaderParamSetter::Create, CANDERA_LINT_REASON_EXPLICITHIDING)
        static GenericShaderParamSetter::SharedPointer Create(); 

        /**
         * Destructs a GenericShaderParamSetter object.
         */
        virtual ~GenericShaderParamSetter() override {}

        /**
         *  Obtain a clone of this parameter setter.
         *  @return a clone of this parameter setter.
         */
        virtual AbstractShaderParamSetter::SharedPointer Clone() const override;

        /**
         * Overrides AbstractShaderParamSetter::Activate
         * @param node           Node on where the shader should be applied.
         * @param appearance     Appearance pass on which the values should be set.
         * @param instanceIndex  The index of the instance these parameters are applied to. A negative index applies parameters immediately.
         * @return     True if shader parameters could be set, False otherwise. Also fails when appearance or appearance's
         *             shader is 0.
         */
        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1735, Candera::GenericShaderParamSetter::Activate, MISRA C++ Rule 8 - 3 - 1 allows the same default arguments for overriding functions)
        virtual bool Activate(Node& node, const MemoryManagement::SharedPointer<Appearance>& appearance, Int instanceIndex = -1) override;

        /**
        *  Indicates whether this setter is suitable for draw call batching using instancing.
        *  @return  True if this setter supports shader parameter handling suitable for instancing, False otherwise.
        */
        virtual bool IsInstancingSupported() const override { return ((!m_isLightsActivationEnabled) || (Light::World == m_lightCoordinateSpace)); }

        /**
         * Retrieves whether uniform shader parameter ModelMatrix4 is calculated and passed to shader or not.
         * See ShaderParamNames::ModelMatrix4 to get uniform string representation used in shader. ModelMatrix4 transforms
         * 4-dimensional position vectors from object space into World space using the node's
         * world transformation matrix retrieved by GetWorldTransform. 
         * This matrix is dedicated to transformations of position vectors, which are scale, rotated and translated.
         * @return Whether shader parameter ShaderParamNames::ModelMatrix4 is calculated and passed to the shader or not. 
         */
        bool IsModelMatrix4Enabled() const { return m_isModelMatrix4Enabled; }

         /**
         * Sets whether uniform shader parameter ModelMatrix4 is calculated and passed to shader or not. 
         * See ShaderParamNames::ModelMatrix4 to get uniform string representation used in shader. ModelMatrix4 transforms
         * 4-dimensional position vectors (mainly vertex positions) from object space into World space using the node's
         * world transformation matrix retrieved by GetWorldTransform. 
         * This matrix is dedicated to transformations of position vectors, which are scale, rotated and translated.
         * @param enableModelMatrix4 Sets whether shader parameter ShaderParamNames::ModelMatrix4 is calculated and passed to the shader or not. 
         */
        void SetModelMatrix4Enabled(bool enableModelMatrix4);

         /**
         * Retrieves whether uniform shader parameter ModelMatrix3 is calculated and passed to shader or not. 
         * See ShaderParamNames::ModelMatrix3 to get uniform string representation used in shader. ModelMatrix3 transforms
         * 3-dimensional position vectors from object space into World space using the node's
         * world transformation matrix retrieved by GetWorldTransform. Additionally this 
         * matrix is inversed and transposed such that the 3x3 matrix is applicable for the vectors.
         * This matrix is dedicated to transformations of position vectors, which are scale, rotated and translated.
         * @return WWhether shader parameter ShaderParamNames::ModelMatrix3 is calculated and passed to the shader or not. 
         */
        bool IsModelMatrix3Enabled() const { return m_isModelMatrix3Enabled; }

         /**
         * Sets whether uniform shader parameter ModelMatrix3 is calculated and passed to shader or not. 
         * See ShaderParamNames::ModelMatrix3 to get uniform string representation used in shader. ModelMatrix3 transforms
         * 3-dimensional position vectors from object space into World space using the node's
         * world transformation matrix retrieved by GetWorldTransform. Additionally this 
         * matrix is inversed and transposed such that the 3x3 matrix is applicable for the vectors. 
         * This matrix is dedicated to transformations of position vectors, which are scale, rotated and translated.
         * @param enableModelMatrix3 Sets whether shader parameter ShaderParamNames::ModelMatrix4 is calculated and passed to the shader or not. 
         */
        void SetModelMatrix3Enabled(bool enableModelMatrix3);

         /**
         * Retrieves whether uniform shader parameter NormalModelMatrix3 is calculated and passed to shader or not. 
         * See ShaderParamNames::NormalModelMatrix3 to get uniform string representation used in shader. NormalModelMatrix3 transforms
         * 3-dimensional direction vectors from object space into World space using the node's
         * world transformation matrix retrieved by GetWorldScale * GetWorldRotation. Hence 
         * the translation is ignored as this matrix applies if only the vector directions are of interest. Additionally this 
         * matrix is inversed and transposed such that the 3x3 matrix is applicable for the vectors.
         * This matrix is dedicated to transformations of direction vectors, here only scale, and rotation applies.
         * @return Whether shader parameter ShaderParamNames::NormalModelMatrix3 is calculated and passed to the shader or not.  
         */
        bool IsNormalModelMatrix3Enabled() const { return m_isNormalModelMatrix3Enabled; }

         /**
         * Sets whether uniform shader parameter NormalModelMatrix3 is calculated and passed to shader or not. 
         * See ShaderParamNames::NormalModelMatrix3 to get uniform string representation used in shader. NormalModelMatrix3 transforms
         * 3-dimensional direction vectors from object space into World space using the node's
         * world transformation matrix retrieved by GetWorldScale * GetWorldRotation. Hence 
         * the translation is ignored as this matrix applies if only the vector directions are of interest. Additionally this 
         * matrix is inversed and transposed such that the 3x3 matrix is applicable for the vectors. 
         * This matrix is dedicated to transformations of direction vectors, here only scale, and rotation applies.
         * @param enableNormalModelMatrix3 Sets whether shader parameter ShaderParamNames::NormalModelMatrix3 is calculated and passed to the shader or not. 
         */
        void SetNormalModelMatrix3Enabled(bool enableNormalModelMatrix3);

         /**
         * Retrieves whether uniform shader parameter ModelViewMatrix4 is calculated and passed to shader or not. 
         * See ShaderParamNames::ModelViewMatrix4 to get uniform string representation used in shader. ModelViewMatrix4 transforms
         * 4-dimensional position vectors from object space into view space using the node's
         * world transformation matrix retrieved by GetWorldTransform and the camera's view 
         * transformation retrieved by GetViewMatrix. 
         * This matrix is dedicated to transformations of position vectors, which are scale, rotated and translated.
         * @return Whether shader parameter ShaderParamNames::ModelViewMatrix4 is calculated and passed to the shader or not. 
         */
        bool IsModelViewMatrix4Enabled() const { return m_isModelViewMatrix4Enabled; }

         /**
         * Sets whether uniform shader parameter ModelViewMatrix4 is calculated and passed to shader or not. 
         * See ShaderParamNames::ModelViewMatrix4 to get uniform string representation used in shader. ModelViewMatrix4 transforms
         * 4-dimensional position vectors from object space into view space using the node's
         * world transformation matrix retrieved by GetWorldTransform and the camera's view 
         * transformation retrieved by GetViewMatrix. 
         * This matrix is dedicated to transformations of position vectors, which are scale, rotated and translated.
         * @param enableModelViewMatrix4 Sets whether shader parameter ShaderParamNames::ModelViewMatrix4 is calculated and passed to the shader or not.  
         */
        void SetModelViewMatrix4Enabled(bool enableModelViewMatrix4);

          /**
          * Retrieves whether uniform shader parameter ModelViewMatrix3 is calculated and passed to shader or not. 
          * See ShaderParamNames::ModelViewMatrix3 to get uniform string representation used in shader. ModelViewMatrix3 transforms
          * 3-dimensional position vectors from object space into view space using the node's
          * world transformation matrix retrieved by GetWorldTransform and the camera's view 
          * transformation retrieved by GetViewMatrix. Additionally this 
          * matrix is inversed and transposed such that the 3x3 matrix is applicable for the vectors. 
          * This matrix is dedicated to transformations of position vectors, which are scale, rotated and translated.
          * @return Whether shader parameter ShaderParamNames::ModelViewMatrix3 is calculated and passed to the shader or not. 
         */
        bool IsModelViewMatrix3Enabled() const { return m_isModelViewMatrix3Enabled; }

         /**
          * Sets whether uniform shader parameter ModelViewMatrix3 is calculated and passed to shader or not. 
          * See ShaderParamNames::ModelViewMatrix3 to get uniform string representation used in shader. ModelViewMatrix3 transforms
          * 3-dimensional position vectors from object space into view space using the node's
          * world transformation matrix retrieved by GetWorldTransform and the camera's view 
          * transformation retrieved by GetViewMatrix. Additionally this 
          * matrix is inversed and transposed such that the 3x3 matrix is applicable for the vectors. 
          * This matrix is dedicated to transformations of position vectors, which are scale, rotated and translated.
          * @param enableModelViewMatrix3 Sets whether shader parameter ShaderParamNames::ModelViewMatrix3 is calculated and passed to the shader or not.  
         */
        void SetModelViewMatrix3Enabled(bool enableModelViewMatrix3);

        /**
         * Retrieves whether uniform shader parameter NormalModelViewMatrix3 is calculated and passed to shader or not. 
         * See ShaderParamNames::NormalModelViewMatrix3 to get uniform string representation used in shader. NormalModelViewMatrix3 transforms
         * 3-dimensional direction vectors from object space into view space using the node's
         * world transformation matrix retrieved by GetWorldScale * GetWorldRotation.and the camera's view 
         * transformation retrieved by GetViewMatrix. Hence the translation is ignored as this matrix applies
         * if only the vector directions are of interest. Additionally this 
         * matrix is inversed and transposed such that the 3x3 matrix is applicable for the vectors.
         * This matrix is dedicated to transformations of direction vectors, here only scale, and rotation applies.
         * @return Whether shader parameter ShaderParamNames::NormalModelViewMatrix3 is calculated and passed to the shader or not. 
         */
        bool IsNormalModelViewMatrix3Enabled() const { return m_isNormalModelViewMatrix3Enabled; }

         /**
         * Sets whether uniform shader parameter NormalModelViewMatrix3 is calculated and passed to shader or not. 
         * See ShaderParamNames::NormalModelViewMatrix3 to get uniform string representation used in shader. NormalModelViewMatrix3 transforms
         * 3-dimensional direction vectors from object space into view space using the node's
         * world transformation matrix retrieved by GetWorldScale * GetWorldRotation and the camera's view 
         * transformation retrieved by GetViewMatrix. Hence the translation is ignored as this matrix applies
         * if only the vector directions are of interest. Additionally this 
         * matrix is inversed and transposed such that the 3x3 matrix is applicable for the vectors.
         * This matrix is dedicated to transformations of direction vectors, here only scale, and rotation applies.
         * @param enableNormalModelViewMatrix3 Sets whether shader parameter ShaderParamNames::NormalModelViewMatrix3 is calculated and passed to the shader or not.  
         */
        void SetNormalModelViewMatrix3Enabled(bool enableNormalModelViewMatrix3);

         /**
         * Retrieves whether uniform shader parameter ModelViewProjectionMatrix4 is calculated and passed to shader or not. 
         * See ShaderParamNames::ModelViewProjectionMatrix4 to get uniform string representation used in shader. ModelViewProjectionMatrix4 transforms
         * 4-dimensional vectors from object space into screen space using the node's
         * world transformation matrix retrieved by GetWorldTransform and the camera's view projection
         * transformation retrieved by GetViewProjectionMatrix. 
         * This matrix is enabled per default as it's needed in most cases to transform vertex coordinates to screen space.
         * @return Whether shader parameter ShaderParamNames::ModelViewProjectionMatrix4 is calculated and passed to the shader or not.  
         */
        bool IsModelViewProjectionMatrix4Enabled() const { return m_isModelViewProjectionMatrix4Enabled; }

         /**
         * Sets whether uniform shader parameter ModelViewProjectionMatrix4 is calculated and passed to shader or not. 
         * See ShaderParamNames::ModelViewProjectionMatrix4 to get uniform string representation used in shader. ModelViewProjectionMatrix4 transforms
         * 4-dimensional vectors from object space into screen space using the node's
         * world transformation matrix retrieved by GetWorldTransform and the camera's view projection
         * transformation retrieved by GetViewProjectionMatrix.
         * This matrix is enabled per default as it's needed in most cases to transform vertex coordinates to screen space.
         * @param enableModelViewProjectionMatrix4 Sets whether shader parameter ShaderParamNames::ModelViewProjectionMatrix4 is calculated and passed to the shader or not.
         */
        void SetModelViewProjectionMatrix4Enabled(bool enableModelViewProjectionMatrix4);

        /**
         * Retrieves whether uniform shader parameter ProjectionMatrix4 is calculated and passed to shader or not.
         * See ShaderParamNames::ProjectionMatrix4 to get uniform string representation used in shader. ProjectionMatrix4 transforms
         * 4-dimensional vectors from view space into screen space using the camera's view projection transformation retrieved
         * by GetProjectionMatrix.
         * This matrix is disabled per default as the ModelViewProjection4 matrix is used in most cases to transform
         * vertex coordinates to screen space.
         * @return Whether shader parameter ShaderParamNames::ProjectionMatrix4 is calculated and passed to the shader or not.
         */
        bool IsProjectionMatrix4Enabled() const { return m_isProjectionMatrix4Enabled; }

        /**
         * Sets whether uniform shader parameter ProjectionMatrix4 is calculated and passed to shader or not. 
         * See ShaderParamNames::ProjectionMatrix4 to get uniform string representation used in shader. ProjectionMatrix4 transforms
         * 4-dimensional vectors from view space into screen space using the camera's view projection transformation retrieved
         * by GetViewProjectionMatrix.
         * This matrix is disabled per default as the ModelViewProjection4 matrix is used in most cases to transform
         * vertex coordinates to screen space.
         * @param enableProjectionMatrix4 Sets whether shader parameter ShaderParamNames::ProjectionMatrix4 is calculated and passed to the shader or not.
         */
        void SetProjectionMatrix4Enabled(bool enableProjectionMatrix4);

        /**
         * Retrieves whether uniform shader parameter CameraLookAtVector is calculated and passed to shader or not. 
         * See ShaderParamNames::CameraLookAtVector to get uniform string representation used in shader. The camera's look at vector describes the direction where
         * the camera is looking at, using a three-component vector.
         * @return Whether shader parameter ShaderParamNames::CameraLookAtVector is calculated and passed to the shader or not.  
         */
        bool IsCameraLookAtVectorEnabled() const { return m_isCameraLookAtVectorEnabled; }

        /**
         * Sets whether uniform shader parameter CameraLookAtVector is calculated and passed to shader or not. 
         * See ShaderParamNames::CameraLookAtVector to get uniform string representation used in shader.
         * The camera's look at vector describes the direction where
         * the camera is looking at, using a three-component vector.
         * @param enableCameraLookAtVector Sets whether shader parameter ShaderParamNames::CameraLookAtVector is calculated and passed to the shader or not. 
         */
        void SetCameraLookAtVectorEnabled(bool enableCameraLookAtVector);

        /**
         * Retrieves whether uniform shader parameter CameraPosition is calculated and passed to shader or not. 
         * See ShaderParamNames::CameraPosition to get uniform string representation used in shader.
         * CameraPosition is passed as three-component vector.
         * @return Whether shader parameter ShaderParamNames::CameraPosition is calculated and passed to the shader or not. 
         */
        bool IsCameraPositionEnabled() const { return m_isCameraPositionEnabled; }

        /**
         * Sets whether uniform shader parameter CameraPosition is calculated and passed to shader or not. 
         * See ShaderParamNames::CameraPosition to get uniform string representation used in shader.
         * CameraPosition is passed as three-component vector and is passed in world space.
         * @param enableCameraPosition Sets whether shader parameter ShaderParamNames::CameraPosition is calculated and passed to the shader or not.
         */
        void SetCameraPositionEnabled(bool enableCameraPosition);

        /**
         * Gets whether activation of light parameters is enabled or not. If enabled then  
         * all parameters needed for light calculation and stored in the active light objects are 
         * passed to the current appearance's shader.
         * This property is enabled per default as light is applied by default mesh shaders.
         * @return Whether shader parameters of lighting are calculated and passed to the shader or not.
         */
        bool IsLightActivationEnabled() const {return m_isLightsActivationEnabled; }

         /**
         * Sets whether activation of light parameters is enabled or not. If enabled then  
         * all parameters needed for light calculation and stored in the active light objects are 
         * passed to the current appearance's shader.
         * This property is enabled per default as light is applied by default mesh shaders.
         * @param enableLightActivation Sets whether shader parameters of lighting are calculated and passed to the shader or not.
         */
        void SetLightActivationEnabled(bool enableLightActivation);

        /**
         * Gets whether activation of material uniforms is enabled or not. If enabled then  
         * all material parameters stored in the current appearances material are 
         * passed to the current appearance's shader.
         * This property is enabled per default as material is normally applied if it's set in the appearance.
         * @return Whether shader parameters of material are passed to the shader or not.
         */
        bool IsMaterialActivationEnabled() const {return m_isMaterialActivationEnabled; }

        /**
         * Sets whether activation of material uniforms is enabled or not. If enabled then  
         * all material parameters stored in the current appearances material are 
         * passed to the current appearance's shader.
         * This property is enabled per default as material is normally applied if it's set in the appearance.
         * @param enableMaterialActivation Sets whether shader parameters of material are passed to the shader or not.
         */
        void SetMaterialActivationEnabled(bool enableMaterialActivation);

        /**
         * Gets whether activation of morphing mesh uniforms is enabled or not.
         * @return enableMorphingMeshActivation Sets whether shader parameters of a morphing mesh are passed to the shader or not.
         */
        bool IsMorphingMeshActivationEnabled() const { return m_isMorphingMeshActivationEnabled; }

        /**
         * Sets whether activation of morphing mesh uniforms is enabled or not. If enabled then
         * morphing mesh uniforms are passed to the current appearance's shader.
         * This property is disabled per default, and needs to be explicitly enabled when used with a morphing mesh.
         * @param enableMorphingMeshActivation Sets whether shader parameters of a morphing mesh are passed to the shader or not.
         */
        void SetMorphingMeshActivationEnabled(bool enableMorphingMeshActivation);

        /**
         * Gets whether activation of point sprite uniforms is enabled or not.
         * @return Whether shader parameters of point sprites are passed to the shader or not.
         */
        bool IsPointSpriteActivationEnabled() const { return m_isPointSpriteActivationEnabled; }

        /**
         * Sets whether activation of point sprite uniforms is enabled or not. If enabled then
         * point sprite uniforms are passed to the current appearance's shader.
         * This property is disabled per default, and needs to be explicitly enabled when used with a point sprite.
         * @param enablePointSpriteActivation Sets whether shader parameters of a point sprite are passed to the shader or not.
         */
        void SetPointSpriteActivationEnabled(bool enablePointSpriteActivation);

        /**
         * Gets whether activation of textures is enabled or not. If enabled then  
         * the appropriate uniforms are set for all textures stored in the current appearance.
         * This property is enabled per default as texturing is normally applied if textures are set on the appearance.
         * @return Whether texture uniforms are passed to the shader or not.
         */
        bool IsTextureActivationEnabled() const { return m_isTextureActivationEnabled; }

        /**
         * Sets whether activation of textures is enabled or not. If enabled then  
         * the appropriate uniforms are set for all textures stored in the current appearance.
         * This property is enabled per default as texturing is normally applied if textures are set on the appearance.
         * @param enableTextureActivation Sets whether texture uniforms are passed to the shader or not.
         */
        void SetTextureActivationEnabled(bool enableTextureActivation);
        
        /**
         * Gets whether activation of a canvas transformation is enabled or not. 
         * If enabled then canvas uniforms are passed to the current appearance's shader.
         * This property is disabled per default, and needs to be explicitly enabled when used with a canvas renderable.
         * @return Whether shader parameters of canvas renderables are passed to the shader or not.
         */
        bool IsCanvasActivationEnabled() const { return m_isCanvasActivationEnabled; }

        /**
        * Sets whether activation of canvas uniforms is enabled or not. If enabled then
        * canvas uniforms are passed to the current appearance's shader.
        * This property is disabled per default, and needs to be explicitly enabled when used with a canvas renderable.
        * @param enableCanvasActivation Sets whether shader parameters of a canvas rendereable are passed to the shader or not.
        */
        void SetCanvasActivationEnabled(bool enableCanvasActivation);

        /**
         * Gets the coordinate space used for lighting calculations.
         * ObjectSpace is the default value.
         * @return lightCoordinateSpace The coordinate space used for lighting calculations to set.
         */
        Light::CoordinateSpace GetLightsCoordinateSpace() const { return m_lightCoordinateSpace; }

         /**
         * Sets the coordinate space used for lighting calculations.
         * ObjectSpace is the default value.
         * @param lightCoordinateSpace The coordinate space used for lighting calculations.
         */
        void SetLightsCoordinateSpace(Light::CoordinateSpace lightCoordinateSpace) {m_lightCoordinateSpace = lightCoordinateSpace; }

        FEATSTD_RTTI_DECLARATION();
        
    protected:
        // Explicit protected Constructor and Copy-Constructor, use Create() to create an instance of this object.
        GenericShaderParamSetter();
        GenericShaderParamSetter(const GenericShaderParamSetter& rhs);
        GenericShaderParamSetter& operator = (const GenericShaderParamSetter& vertexBuffer);

    private:
        bool m_isModelMatrix4Enabled;
        bool m_isModelMatrix3Enabled;
        bool m_isNormalModelMatrix3Enabled;
        bool m_isModelViewMatrix4Enabled;
        bool m_isModelViewMatrix3Enabled;
        bool m_isNormalModelViewMatrix3Enabled;
        bool m_isModelViewProjectionMatrix4Enabled;
        bool m_isProjectionMatrix4Enabled;
        bool m_isCameraLookAtVectorEnabled;
        bool m_isCameraPositionEnabled;
        bool m_isLightsActivationEnabled;
        bool m_isMaterialActivationEnabled;
        bool m_isMorphingMeshActivationEnabled;
        bool m_isPointSpriteActivationEnabled;
        bool m_isTextureActivationEnabled;
        bool m_isCanvasActivationEnabled;
        Light::CoordinateSpace m_lightCoordinateSpace;

        typedef bool(GenericShaderParamSetter::*ActivateFn)(const Node& node, const Appearance& appearance, Shader& shader, Int instanceIndex) const;

        typedef Internal::Vector<ActivateFn> AutoUniformContainer;
        AutoUniformContainer m_autoUniforms;

        void SetAutoUniformEnabled(bool& currentEnableState, bool newEnabledState, ActivateFn activateFunction);

        bool ActivateModelMatrix3(const Node& node, const Appearance& appearance, Shader& shader, Int instanceIndex) const;
        bool ActivateModelMatrix4(const Node& node, const Appearance& appearance, Shader& shader, Int instanceIndex) const;
        bool ActivateNormalModelMatrix(const Node& node, const Appearance& appearance, Shader& shader, Int instanceIndex) const;
        bool ActivateModelViewMatrix3(const Node& node, const Appearance& appearance, Shader& shader, Int instanceIndex) const;
        bool ActivateModelViewMatrix4(const Node& node, const Appearance& appearance, Shader& shader, Int instanceIndex) const;
        bool ActivateNormalModelViewMatrix(const Node& node, const Appearance& appearance, Shader& shader, Int instanceIndex) const;
        bool ActivateModelViewProjectionMatrix(const Node& node, const Appearance& appearance, Shader& shader, Int instanceIndex) const;
        bool ActivateProjectionMatrix(const Node& node, const Appearance& appearance, Shader& shader, Int instanceIndex) const;
        bool ActivateCameraLookAtVector(const Node& node, const Appearance& appearance, Shader& shader, Int instanceIndex) const;
        bool ActivateCameraPosition(const Node& node, const Appearance& appearance, Shader& shader, Int instanceIndex) const;
        bool ActivateLights(const Node& node, const Appearance& appearance, Shader& shader, Int instanceIndex) const;
        bool ActivateMaterial(const Node& node, const Appearance& appearance, Shader& shader, Int instanceIndex) const;
        bool ActivateMorphingMesh(const Node& node, const Appearance& appearance, Shader& shader, Int instanceIndex) const;
        bool ActivatePointSprite(const Node& node, const Appearance& appearance, Shader& shader, Int instanceIndex) const;
        bool ActivateTextures(const Node& node, const Appearance& appearance, Shader& shader, Int instanceIndex) const;
        bool ActivateCanvasRenderable(const Node& node, const Appearance& appearance, Shader& shader, Int instanceIndex) const;

        static const GenericShaderParamSetter::ActivateFn* GetActivationFunctionList();

        static Shader::UniformCacheHandle s_uniformCacheHandle;
};

/** @} */ // end of ShaderParamSetters3D

}// namespace

#endif    // Candera_TransAnisotropicLightShaderParamSetter_h

