//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Candera_AbstractShaderParamSetter_h)
#define Candera_AbstractShaderParamSetter_h

#include <Candera/EngineBase/Common/CanderaObject.h>
#include <Candera/System/MemoryManagement/SharedPointer.h>

namespace Candera {
/** @addtogroup ShaderParamSetters3D
 *  @{
 */

//Forward declarations.
class Node;
class Appearance;

/**
 * @brief AbstractShaderParamSetter provides an interface to pass shader parameters like uniforms and constant attributes to a Shader.
 *
 *  Note: The Shader is assumed to be already the currently active GPU program when ShaderParamSetter is called to do its work.
 */
class AbstractShaderParamSetter : public CanderaObject
{
        FEATSTD_TYPEDEF_BASE(CanderaObject);

    public:
        FEATSTD_TYPEDEF_SHARED_POINTER(AbstractShaderParamSetter);

        /**
         *  Constructor.
         */
        AbstractShaderParamSetter() {}

        /**
         *  Destructor.
         */
        virtual ~AbstractShaderParamSetter() override {}

        /**
         *  Obtain a clone of this parameter setter.
         *  @return a clone of this parameter setter.
         */
        virtual SharedPointer Clone() const = 0;

        /**
         *  Activates the shader parameters of this ShaderParamSetter object.
         *  @param node           Node on where the shader should be applied.
         *  @param appearance     Appearance pass on which the values should be set.
         *  @param instanceIndex  The index of the instance these parameters are applied to. A negative index applies parameters immediately.
         *  @return     True if shader parameters could be set, False otherwise. Also fails when appearance or appearance's
         *              shader is 0.
         */
        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1735, Candera::AbstractShaderParamSetter::Activate, MISRA C++ Rule 8-3-1 allows the same default arguments for overriding functions)
        virtual bool Activate(Node& node, const MemoryManagement::SharedPointer<Appearance>& appearance, Int instanceIndex = -1) = 0;

        /**
         *  Indicates whether this setter is suitable for draw call batching using instancing.
         *  @return  True if this setter supports shader parameter handling suitable for instancing, False otherwise.
         */
        virtual bool IsInstancingSupported() const { return false; }

        FEATSTD_RTTI_DECLARATION();

    protected:
        AbstractShaderParamSetter(const AbstractShaderParamSetter& rhs) : Base(rhs) {}

    private:
        AbstractShaderParamSetter& operator =(const AbstractShaderParamSetter& rhs);
        // Make this class manageable by MemoryManagement::SharedPointer
        CANDERA_SHARED_POINTER_DECLARATION();
};

 /** @} */ // end of ShaderParamSetters3D
} // namespace Candera

#endif// Candera_ShaderParamSetter_h
