//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Candera_RenderStateOrderCriterion_h)
#define Candera_RenderStateOrderCriterion_h

#include <Candera/Engine3D/RenderOrder/RenderOrderCriterion.h>

namespace Candera {

/** @addtogroup RenderOrder3D
 *  @{
 */

// forward declaration.
class Node;

/**
 * RenderStateOrderCriterion implements a proof of concept OrderCriterion used for batching nodes to minimize
 *  render state changes.
 *  
 * RenderStateOrderCriterion takes in consideration the Node's Shader, Texture and RenderMode, whose changes 
 *  are considered the most expensive. An OrderCriterionValue of 32 bits is constructed as follows: 
 * - 5 bits (msb): hash value representing the Node's Shader Handle.
 * - 6 bits: hash value representing the Node's Texture Handle.
 * - 1 bit:  RenderMode blending enabled flag.
 * - 10 bits: RenderMode alpha blend mode.
 * - 1 bit: RenderMode depth write enabled flag.
 * - 1 bit: RenderMode depth test enabled flag.
 * - 1 bit: RenderMode stencil test enabled flag.
 * - 4 bits: RenderMode color write (RGBA) enabled flags.
 * - 1 bit: RenderMode winding mode.
 * - 2 bit (lsb): RenderMode culling mode.
 *  
 * After sorting all the nodes using the OrderCriterionValue, they will be rendered in batches of nodes with the same Shader.
 *  Inside a batch of nodes with the same Shader, the rendering will be done in batches of nodes with the same Texture. 
 *  Inside a batch of nodes with the same Shader and Texture, the rendering will be done in batches of nodes with the same
 *  blending enabled value. The batching continues till the least expensive state change (CullingMode).
 *    
 * Notes: 
 * - Usage of this OrderCriterion does not guarantee best batching algorithm, which should be implemented
 *  after benchmarking the target render device's state change costs.
 * - RenderStateOrderCriterion groups the nodes only by their first Appearance and Texture. Nodes with 
 *  multi-pass rendering (MultiPassAppearance), or with multiple Texture units, most probably will break the 
 *  batches and cancel some performance gains.
 * - CANDERA_RENDER_STATE_CACHING_ENABLED should be defined to use Candera's RenderStateCache mechanism.
 */
class RenderStateOrderCriterion : public RenderOrderCriterion
{
    FEATSTD_TYPEDEF_BASE(RenderOrderCriterion);

    public:
        /**
         *  Constructs a RenderStateOrderCriterion object.
         */
        RenderStateOrderCriterion() {}

        /**
         *  Destructs a RenderStateOrderCriterion object.
         */
        virtual ~RenderStateOrderCriterion() override {}
        
        /**
          *  Overrides OrderCriterion::PrepareRenderOrderCriterionValue. 
          *  
          *  A 32 bit unsigned integer OrderCriterionValue is computed for each node based on its render states.
          *   The OrderCriterionValue will be used for sorting the nodes, thus minimizing the most expensive 
          *   render state changes.
          *   
          * @param n Node reference.
          */
        virtual void PrepareRenderOrderCriterionValue(Node& n) const override;

        /**
         *  Compares two Node's precomputed OrderCriterionValue.
         *  @param a First Node whose OrderCriterionValue is compared to Node b.
         *  @param b Second Node whose OrderCriterionValue is compared to Node a.
         *  @return true if first Node's OrderCriterionValue is less than second Node's OrderCriterionValue.
         */
        virtual bool IsBefore(const Node& a, const Node& b) const override;

        /**
         *  Calculates the order criterion values (i.e. sorting keys) from the rank for the given nodes.
         *  @param nodeContainer     The container of nodes to compute the sorting keys for.
         *  @param sortingContainer  The container to store the sorting keys and node indices.
         *  @param nodeContainerSize The number of nodes in use in the nodeContainer. This can be
         *                           smaller than the actual size of the container
         *  @return True if all order criterion values could be prepared, False otherwise.
         */
        virtual bool PrepareRenderOrderCriterionValues(Internal::RenderOrderBin::NodeContainer& nodeContainer,
            Internal::RenderOrderBin::SortingContainer& sortingContainer, const SizeType nodeContainerSize) const override;

        /**
         *  Sort the container of the RenderOrderBin
         *  @param sortingContainer  The container to be sorted.
         *  @param containerSize     The size of the container to be sorted. This can be smaller than
         *                           the actual size of the container to sort a subsection.
         */
        virtual void Sort(Internal::RenderOrderBin::SortingContainer& sortingContainer, const SizeType containerSize) const override;

        /**
         *  Determine if the render order has changed from the previous pass to the current one.
         *  If the order has not changed, the render order bin will use the result of the previous pass,
         *  instead of sorting the current pass.
         *  @param currentBin   The sorting bin of the previous pass.
         *  @param previousBin  The sorting bin of the current pass.
         *  @return True, if the bins are different or the order has changed. False, otherwise.
         */
        virtual bool HasOrderChanged(const Internal::RenderOrderBin::SortingBin& currentBin,
            const Internal::RenderOrderBin::SortingBin& previousBin) const;

        FEATSTD_RTTI_DECLARATION();
    
    private:
        union RenderState{
            struct {
                UInt m_rmCullingMode                : 2;
                UInt m_rmWindingMode                : 1;
                UInt m_rmAWriteEnabled              : 1;
                UInt m_rmBWriteEnabled              : 1;
                UInt m_rmGWriteEnabled              : 1;
                UInt m_rmRWriteEnabled              : 1;
                UInt m_rmStencilTestEnabled         : 1;
                UInt m_rmDepthTestEnabled           : 1;
                UInt m_rmDepthWriteEnabled          : 1;
                UInt m_rmAlphaBlendModeDstFactor    : 4;
                UInt m_rmAlphaBlendModeOperation    : 2;
                UInt m_rmAlphaBlendModeSrcFactor    : 4;
                UInt m_rmBlendingEnabled            : 1;
                UInt m_textureHash                  : 6;
                UInt m_shaderHash                   : 5;
            } m_bits;
            UInt32 m_value;
        };
};

 /** @} */ // end of RenderOrder3D

}

#endif // Candera_RenderStateOrderCriterion_h
