//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_RENDERORDERBIN_H)
#define CANDERA_RENDERORDERBIN_H

#include <Candera/Environment.h>
#include <Candera/Engine3D/RenderOrder/OrderCriterionValue.h>
#include <Candera/EngineBase/Common/CanderaObject.h>
#include <Candera/System/Container/Vector.h>

namespace Candera {

/** @addtogroup RenderOrder3D
 *  @{
 */

// forward declaration.
class Node;
class OrderCriterion;

/**
 *  @brief  class RenderOrderBin defines a set of Nodes to be rendered. After the entire scene has been traversed and all Nodes
 *            have been assigned to RenderOrderBins, then the RenderOrderBins are rendered in order according to their rank in an
 *          ascending order. Within each RenderOrderBin, the Nodes are sorted according to RenderOrderBin's associated OrderCriterion.
 */

namespace Internal {

class RenderOrderBin : public CanderaObject {
    FEATSTD_TYPEDEF_BASE(CanderaObject);

    public:

        /**
         *  Creates a RenderOrderBin with default values.
         *  @param maxNodeCount Maximum number of nodes to be stored.
         */
        RenderOrderBin(Int32 maxNodeCount);

        /**
         *  Copy constructor that creates a shallow clone.
         *  @param src The other RenderOrderBin to clone from.
         */
        RenderOrderBin(const RenderOrderBin& src);

        /**
         *  Destructor
         */
        virtual ~RenderOrderBin() override;

        /**
         *  Set the rank, which defines the ascending render order of all RenderOrderBins stored in RenderOrder.
         *  @param rank Rank to be set.
         */
        void SetRank(Int32 rank) { m_rank = rank; }

        /**
         *  Retrieves the rank, which defines the ascending render order of all RenderOrderBins stored in RenderOrder.
         *  @return The rank.
         */
        Int32 GetRank() const { return m_rank; }

        /**
         *  Set the hash of the name of the bin.
         *  @param rank The hash of the name of the bin to be set.
         */
        void SetNameHash(UInt32 nameHash) { m_nameHash = nameHash; }

        /**
         *  Set the hash of the name of the bin.
         *  @return The hash of the name of the bin.
         */
        UInt32 GetNameHash() const { return m_nameHash; }

        /**
         *  Define if this RenderOrderBin shall be sorted or not according to associated OrderCriterion (For more details see
         *  function Sort).
         *  @param enable    true enables sorting, false disables sorting.
         */
        void SetSortingEnabled(bool enable) { m_isSortingEnabled = enable; }

        /**
         *  Retrieves whether sorting the RenderOrderBin is enabled or not.
         *  @return True, if the sorting is enabled, false otherwise.
         */
        bool IsSortingEnabled() const { return m_isSortingEnabled; }

        /**
         *  Set the order criterion of Nodes in this bin, and therefore the sorting order.
         *  @param orderCriterion
         */
        void SetOrderCriterion(const OrderCriterion* orderCriterion);

        /**
         *  Retrieves the order criterion of Nodes in this bin.
         *  @return The order criterion of Nodes in this bin.
         */
        const OrderCriterion* GetOrderCriterion() const { return m_orderCriterion; }

        /**
         *  Removes all Node associations in this RenderOrderBin.
         */
        void Clear();

        /**
         *  Adds a Node association to this RenderOrderBin.
         *  @param node Node to be associated to this RenderOrderBin.
         *  @return True if node was added to this RenderOrderBin.
         */
        bool AddNode(Node* node); 

        /**
         *  If sorting is enabled (see SetSortingEnabled) then this RenderOrderBin will be sorted according to the
         *  concrete OrderCriterion associated.
         */
        void Sort();

        /**
         *  Returns number of Nodes stored in this RenderOrderBin.
         *  @return Number of Nodes stored in this RenderOrderBin.
         */
        SizeType GetSize() const { return m_bins[m_currentBinIndex].GetSize(); }

        /**
         *  Returns the maximum number of Nodes to be stored in this.
         *  RenderOrderBin.
         *  @return The maximum number of Nodes.
         */
        SizeType GetMaxSize() const { return m_bins[m_currentBinIndex].GetNodeContainer().GetCapacity(); }

        /**
         *  Retrieve if this RenderOrderBin is empty.
         *  @return true if this RenderOrderBin is empty and false if not.
         */
        bool IsEmpty() const { return GetSize() == 0; }

        /**
         *  Retrieve Node by index.
         *  @param index Index to get node from array.
         *  @return Node by index.
         */
        Node* GetNode(Int32 index) const;

        /**
         *  The SortingElement structure contains the order criterion value (i.e. sorting key) and the index to
         *  the node in the bin.
         */
        struct SortingElement {
            public:
                SortingElement(UInt32 key, UInt16 indexOfNode) : m_key(key), m_secondaryKey(0), m_indexOfNode(indexOfNode) {}
                SortingElement(UInt32 key, UInt16 secondaryKey, UInt16 indexOfNode) : m_key(key), m_secondaryKey(secondaryKey), m_indexOfNode(indexOfNode) {}
                SortingElement(Float key, UInt16 indexOfNode) : m_key(key), m_secondaryKey(0), m_indexOfNode(indexOfNode) {}
                bool IsBeforeFloat(const SortingElement& element) const { return (m_key.GetFloat() < element.m_key.GetFloat()); }
                bool IsBeforeUInt32(const SortingElement& element) const { return (m_key.GetUInt32() < element.m_key.GetUInt32()); }
                bool IsBeforeInt32(const SortingElement& element) const { return (GetInt32() < element.GetInt32()); }
                UInt16 GetIndexOfNode() const { return m_indexOfNode; }
                Float GetFloat() const { return m_key.GetFloat(); }
                UInt32 GetUInt32() const { return m_key.GetUInt32(); }
                Int32 GetInt32() const { return static_cast<Int32>(m_key.GetUInt32()); }
                UInt16 GetSecondaryKey() const { return m_secondaryKey; }

            private:
                OrderCriterionValue m_key;
                UInt16 m_secondaryKey;
                UInt16 m_indexOfNode;
        };

        typedef Candera::Internal::Vector<SortingElement> SortingContainer;
        typedef Candera::Internal::Vector<Node*> NodeContainer;

        /**
         *  The SortingBin structure combines the data necessary to perform double buffering on bins internally,
         *  as well as exposing a small interface to be used by render order criterion classes.
         */
        struct SortingBin {
            public:
                SortingBin() : m_nodeCount(0) {}

                void Clear() { m_nodeCount = 0; }
                bool AddNode(Node* node) {
                    if (FeatStd::Internal::NumericConversion<Int32>(m_nodeContainer.Size()) <= m_nodeCount) {
                        const bool addOk = m_nodeContainer.Add(node);
                        if (addOk) {
                            ++m_nodeCount;
                        }
                        return addOk;
                    }
                    else
                    {
                        m_nodeContainer[m_nodeCount] = node;
                        ++m_nodeCount;
                        return true;
                    }
                }

                Int32 GetSize() const { return m_nodeCount; }
                const NodeContainer& GetNodeContainer() const { return m_nodeContainer; }
                NodeContainer& GetNodeContainer() { return m_nodeContainer; }
                const SortingContainer& GetSortingContainer() const { return m_sortingContainer; }
                SortingContainer& GetSortingContainer() { return m_sortingContainer; }

            private:
                Int32 m_nodeCount;
                NodeContainer m_nodeContainer;    // Container for Nodes maintained by this RenderOrderBin.
                SortingContainer m_sortingContainer;
        };

    private:
        void SwapSortingBins() { m_currentBinIndex = 1 - m_currentBinIndex; }

        UInt32 m_nameHash;
        Int32 m_rank;                // Rank of this bin within ascending render order of bins.
        const OrderCriterion* m_orderCriterion; // Defines the concrete render order of Nodes in this RenderOrderBin.

        SortingBin m_bins[2];
        UInt m_currentBinIndex;

        bool m_isSortingEnabled;        // Defines if sorting is enabled or not.
        bool m_isRenderOrderCriterionAndSorted;  // Indicates if m_orderCriterion is a RenderOrderCriterion and the bin has been sorted.
};

 /** @} */ // end of RenderOrder3D

}

}

#endif // CANDERA_RENDERORDERBIN_H
