//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_RenderOrder_H)
#define CANDERA_RenderOrder_H

#include <Candera/Engine3D/RenderOrder/AbstractRenderOrder.h>
#include <Candera/Engine3D/RenderOrder/RenderOrderBin.h>
#include <Candera/System/Container/Vector.h>

namespace Candera {

/** @addtogroup RenderOrder3D
 *  @{
 */

/**
 *  @brief Defines the order(sequence) in which nodes are rendered. Collaborates with Renderer.
 *  Must cover functional correctness (translucent/transparent objects after opaque/solid objects) as well as
 *  optimization-related orderings (sorting by state, defining a background bin for rendering without depth checks).
 *  Nodes are assigned to bins; all nodes in an "earlier" bin are rendered before all nodes in a "later" bin.
 *  Two default bins are defined and must not be deleted. The opaque bin, which is sorted from front to back (DistanceToCameraOrderCriterion),
 *  and the transparent bin, which is sorted from back to front (ReverseDistanceToCameraOrderCriterion). If a Node has no specific
 *  assignment to a render order bin, it will be automatically assigned by Candera either to opaque or transparent bin, according to its
 *  RenderMode::IsAlphaBlendingEnabled setting.
 */

class RenderOrder: public AbstractRenderOrder {
    FEATSTD_TYPEDEF_BASE(AbstractRenderOrder);

    public:
        static const Char* const OpaqueBinName;         ///< Name of default opaque bin.
        static const Char* const TransparentBinName;    ///< Name of default transparent bin.

        enum {
            DefaultOpaqueBinRank = 10,            ///< Default rank for Opaque bin.
            DefaultTransparentBinRank= 20         ///< Default rank for Transparent bin.
        };

        /**
         *  Creates a RenderOrder with following two default bins:
         *  Opaque bin with default rank "10" sorted from "Front to Back" (= DistanceToCameraOrderCriterion).
         *  Transparent bin with default rank "20" sorted from "Back to Front" (= ReverseDistanceToCameraOrderCriterion).
         *  Use Dispose() to delete the instance.
         *  @param opaqueMaxNodeCount defines the max. number of Nodes that shall be stored in the "Opaque" render order bin.
         *  @param transparentMaxNodeCount defines the max. number of Nodes that shall be stored in the "Transparent" render order bin.
         *  @note The max values are used for presizing the container.
         *  @return Pointer to the created RenderOrder object.
         */
        static RenderOrder* Create(Int32 opaqueMaxNodeCount, Int32 transparentMaxNodeCount);

        /**
         *  Destructs this Render Order object.
         */
        virtual ~RenderOrder() override;

        /**
         *  Creates a object RenderOrderBin with the name given, a certain rank and with the size of the max node count specified.
         *  @param binName defines the name of the bin.
         *  @param rank defines the concrete render order of this bin. Bins with a lower rank are rendered before bins with higher ranks.
         *  @param maxNodeCount defines the max. number of Nodes that can be stored in this bin.
         *  @return true if bin creation was successful and false if not (e.g. if bin with same name already exists).
         */
        bool CreateBin(const Char* binName, Int32 rank, Int32 maxNodeCount);

        /**
         *  Deletes the bin with the name given.
         *  The default bins "Opaque" and "Transparent" cannot be deleted.
         *  @param binName Name of the bin to be deleted.
         *  @return true if bin could be deleted and false if bin could not be deleted, because bin was either not found or
         *  bin's name is "Opaque" or "Transparent".
         */
        bool DeleteBin(const Char* binName);

        /**
         *  Set rank of bin for the bin name given.
         *  @param binName defines the name of bin, which rank shall be set.
         *  @param rank defines the render order of bin, whereas bins with lower ranks are rendered before bins with higher ranks.
         */
        void SetBinRank(const Char* binName, Int32 rank);

        /**
         *  Retrieves the bin rank for the given bin name.
         *  @param binName defines the name of bin, which rank shall be set.
         *  @return The bin rank for the given bin name.
         */
        Int32 GetBinRank(const Char* binName) const;

        /**
         *  Define if this bin's render order shall be sorted or not.
         *  @param binName defines the name of bin.
         *  @param enable defines if sorting is enabled for this bin (true) or not (false).
         */
        void SetBinSortingEnabled(const Char* binName, bool enable);

        /**
         *  Retrieves whether sorting the bin's render order is enabled or not.
         *  @param binName defines the name of bin.
         *  @return True if sorting is enabled, false otherwise.
         */
        bool IsBinSortingEnabled(const Char* binName) const;

        /**
         *  Set bin order criterion to be used by this bin. This bins render order is sorted according to the OrderCriterion set.
         *  @param binName defines the name of bin.
         *  @param orderCriterion defines the concrete OrderCriterion utilized to sort this bin.
         */
        void SetBinOrderCriterion(const Char* binName, const OrderCriterion* orderCriterion);

        /**
         *  Retrieves the bin order criterion for the given bin name.
         *  @param binName defines the name of bin.
         *  @return The bin order criterion.
         */
        const OrderCriterion* GetBinOrderCriterion(const Char* binName) const;

        // overridden functions

        /**
         *  Set iterator of this RenderOrder to first Node in the first bin.
         */
        virtual void IterationBegin() override;

        /**
         *  Returns if there are more Nodes to retrieve. This is true, as long as the last Node in the last bin
         *  has not been retrieved.
         *  @return True if there are more Nodes to retrieve.
         */
        virtual bool IterationHasMoreNodes() override;

        /**
         *  Move iterator to next Node in this render order.
         */
        virtual void IterationMoveToNextNode() override;

        /**
         *  Retrieve the Node the iterator is pointing to.
         *  @return The Node the iterator is pointing to.
         */
        virtual Node* IterationGetNode() override;

        /**
         *  Assign the given Node to a bin defined in class Node. For more details see function Node::SetRenderOrderBinAssignment.
         *  @param node The node to be assigned to the bin.
         *  @return True if bin was assigned successfully, false if node is 0, node cannot be rendered or assigned bin doesn't exist.
         */
        virtual bool AssignNodeToBin(Node* node) override;

        /**
         *  Assign the given Node to the first bin. The given Node has be verified in the same way as AssignNodeToBin does, first.
         *  This is be used to assign nodes from one RenderOrder to another without the need to re-verify each node.
         *  @param node The node to be assigned to the bin.
         *  @return True if bin was assigned successfully, false otherwise.
         */
        virtual bool AssignVerifiedNodeToBin(Node* node) override;

        /**
         *  Sort all bins of this RenderOrder that have sorting enabled. For more details see function RenderOrder::SetBinSortingEnabled.
         */
        virtual void SortBins() override;

        /**
         *  Clear all bins applied to this render order.
         */
        virtual void Clear() override;

        /**
         *  Creates a clone of this RenderOrder and it's RenderOrderBins.
         *  @return A pointer to the clone.
         */
        virtual RenderOrder* Clone() const override;

        /**
         *  Destroys a class instance
         */
        virtual void Dispose() override ;

    protected:
        // Explicit protected Constructor and Copy-Constructor, use Create() to create an instance of this object.
        RenderOrder();
        RenderOrder(Int32 initBinCount, Int32 opaqueMaxNodeCount, Int32 transparentMaxNodeCount);
        RenderOrder(const RenderOrder& src);
        RenderOrder& operator = (const RenderOrder& rhs);

    private:
        Int32 m_binIndex;       // Current index of bin.
        Int32 m_nodeIndex;      // Current index of node.

        Internal::Vector<Internal::RenderOrderBin*> m_binContainer;    // Container of RenderOrderBins.

        Internal::RenderOrderBin m_opaqueBin;      // RenderOrderBin to store opaque nodes.
        Internal::RenderOrderBin m_transparentBin; // RenderOrderBin to store transparent nodes.

        bool m_hasNameHashCollisions; // Flag if the hashes of the names of the bins collide.

        /**
         *  Adds a pointer to a RenderOrderBin into m_binContainer at the correct location based on it's rank.
         *  Note that this function does not check for duplicate bin names and does not explicitly resize m_binContainer.
         *  @param bin A reference to the RenderOrderBin to add.
         *  @return The index of the bin in the bin container, or value -1 if the bin could not be added.
         */
        Int32 InsertBin(Internal::RenderOrderBin* bin);

        /**
         *  Find bin in this RenderOrder by name.
         *  @param binName Name of the bin to look for.
         *  @return the index of the bin found or value -1, if bin was not found.
         */
        Int32 FindBin(const Char* binName) const;

        /**
         *  Find bin in this RenderOrder by name.
         *  @param binName Name of the bin to look for.
         *  @param binNameHash Hash of the name of the bin to look for.
         *  @return the index of the bin found or value -1, if bin was not found.
         */
        Int32 FindBin(const Char* binName, UInt32 binNameHash) const;

        /**
         *  Initializes size of m_binContainer with the given size or
         *  extends it if size is greater than the size of an already existing m_binContainer.
         *  @param size Size to set m_binContainer to.
         *  @return True if reserving size of the vector succeeded or vector already has the right size, false if reserving didn't succeed.
         */
        bool ResizeBinContainer(SizeType size);
};

 /** @} */ // end of RenderOrder3D

} // namespace Candera

#endif    // CANDERA_RenderOrder_H

