//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "RenderOrder.h"
#include <Candera/Engine3D/Core/Appearance.h>
#include <Candera/Engine3D/Core/Camera.h>
#include <Candera/Engine3D/Core/Node.h>
#include <Candera/Engine3D/Core/RenderMode.h>
#include <Candera/Engine3D/Core/Renderer.h>
#include <Candera/Engine3D/RenderOrder/DistanceToCameraOrderCriterion.h>
#include <Candera/Engine3D/RenderOrder/OrderCriterion.h>
#include <Candera/System/Diagnostics/Log.h>
#include <CanderaPlatform/Device/Common/Base/RenderDevice.h>
#include <FeatStd/Util/Hash.h>

namespace Candera {
    using namespace Diagnostics;
    using namespace Candera::Internal;

    FEATSTD_LOG_SET_REALM(LogRealm::CanderaEngine3D);

static const Int32 c_MaxBinCount = 10;

/**
 * Helper function to sort RenderOrderBins by their rank applied.
 */
bool IsRankLess(const RenderOrderBin* lhs, const RenderOrderBin* rhs)
{
    return lhs->GetRank() < rhs->GetRank();
}

// Names for default Opaque and Transparent bins.
const Char* const RenderOrder::OpaqueBinName = "Opaque";
const Char* const RenderOrder::TransparentBinName = "Transparent";

RenderOrder::RenderOrder() :
    Base(),
    m_binIndex(0),
    m_nodeIndex(0),
    m_binContainer(),
    m_opaqueBin(0),
    m_transparentBin(0),
    m_hasNameHashCollisions(false)
{
}

// Create stateless order criteria for opaque and transparent bin.
namespace {
DistanceToCameraOrderCriterion* GetCameraCriterion() {
    static DistanceToCameraOrderCriterion cameraCriterion;
    return &cameraCriterion;
}
ReverseDistanceToCameraOrderCriterion* GetReverseCameraCriterion() {
    static ReverseDistanceToCameraOrderCriterion reverseCameraCriterion;
    return &reverseCameraCriterion;
}
}

RenderOrder::RenderOrder(Int32 initBinCount, Int32 opaqueMaxNodeCount, Int32 transparentMaxNodeCount) :
    Base(),
    m_binIndex(0),
    m_nodeIndex(0),
    m_binContainer(),
    m_opaqueBin(opaqueMaxNodeCount),
    m_transparentBin(transparentMaxNodeCount),
    m_hasNameHashCollisions(false)
{
    // Container requires at least a size of 2 to add the opaque and transparent bins.
    initBinCount = Math::Maximum<Int32>(initBinCount, 2);

    bool success = ResizeBinContainer(initBinCount);

    if (!success) {
        FEATSTD_LOG_ERROR("Resize of bin container failed");
        return;
    }
    FEATSTD_LINT_CURRENT_SCOPE(1938, "Global variables are not being modified.");

    // Initialize opaque bin.
    m_opaqueBin.SetName(OpaqueBinName);
    m_opaqueBin.SetRank(DefaultOpaqueBinRank);
    m_opaqueBin.SetOrderCriterion(GetCameraCriterion());

    // Initialize transparent bin.
    m_transparentBin.SetName(TransparentBinName);
    m_transparentBin.SetRank(DefaultTransparentBinRank);
    m_transparentBin.SetOrderCriterion(GetReverseCameraCriterion());

    // Insert bins into container at their correct sorted position.
    static_cast<void>(InsertBin(&m_opaqueBin));
    static_cast<void>(InsertBin(&m_transparentBin));
}

RenderOrder::RenderOrder(const RenderOrder& src) :
    Base(src),
    m_binIndex(0),
    m_nodeIndex(0),
    m_binContainer(),
    m_opaqueBin(src.m_opaqueBin),
    m_transparentBin(src.m_transparentBin),
    m_hasNameHashCollisions(src.m_hasNameHashCollisions)
{
    static_cast<void>(ResizeBinContainer(src.m_binContainer.GetCapacity()));
    FEATSTD_LINT_CURRENT_SCOPE(1938, "Global variables are not being modified.");

    static_cast<void>(InsertBin(&m_opaqueBin));
    static_cast<void>(InsertBin(&m_transparentBin));

    for (UInt i = 0; i < static_cast<UInt>(src.m_binContainer.Size()); ++i) {

        RenderOrderBin* bin = src.m_binContainer[i];
        if ((bin != 0) && (bin != &src.m_opaqueBin) && (bin != &src.m_transparentBin)) {
            // Only copy custom bins, copies of the opaque and transparent bins have already been made.
            RenderOrderBin* newBin = FEATSTD_NEW(RenderOrderBin)(*bin);
            if (newBin != 0) {
                if (InsertBin(newBin) < 0) {
                    FEATSTD_DELETE(newBin);
                }
            }
            else {
                FEATSTD_LOG_ERROR("Create bin object failed for %s.", bin->GetName());
            }
        }
    }

}

RenderOrder* RenderOrder::Create(Int32 opaqueMaxNodeCount, Int32 transparentMaxNodeCount)
{
    return FEATSTD_NEW(RenderOrder)(c_MaxBinCount, opaqueMaxNodeCount, transparentMaxNodeCount);
}

RenderOrder* RenderOrder::Clone() const
{
    return FEATSTD_NEW(RenderOrder)(*this);
}

void RenderOrder::Dispose()
{
    FEATSTD_DELETE(this);
}

RenderOrder::~RenderOrder()
{
    for (UInt i = 0; i < static_cast<UInt>(m_binContainer.Size()); ++i) {
        if ((m_binContainer[i] != &m_opaqueBin) && (m_binContainer[i] != &m_transparentBin)) {
            FEATSTD_DELETE(m_binContainer[i]);
        }
    }
}

bool RenderOrder::CreateBin(const Char* binName, Int32 rank, Int32 maxNodeCount)
{
    CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(429, CANDERA_LINT_REASON_SHAREDPOINTER)

    if (FindBin(binName) >= 0) {
        //bin with same name already exists.
        return false;
    }

    if (m_binContainer.Size() == m_binContainer.GetCapacity()) {
        if (!ResizeBinContainer(m_binContainer.Size() + 1)) {
            FEATSTD_LOG_ERROR("Resize of bin container failed for %s.", binName);
            return false;
        }
    }

    RenderOrderBin* newBin = FEATSTD_NEW(RenderOrderBin)(maxNodeCount);
    if (newBin == 0) {
        FEATSTD_LOG_ERROR("Create bin object failed for %s.", binName);
        return false;
    }

    newBin->SetName(binName);
    newBin->SetRank(rank);

    if (InsertBin(newBin) < 0) {
        FEATSTD_DELETE(newBin);
        return false;
    }

    return true;
}

bool RenderOrder::DeleteBin(const Char* binName)
{
    // Opaque and Transparent bin must not be deleted
    if ((StringPlatform::CompareStrings(binName, OpaqueBinName) == 0) || (StringPlatform::CompareStrings(binName, TransparentBinName) == 0))
    {
        return false;
    }

    const Int32 idx = FindBin(binName);
    if (idx < 0) {
        return false;
    }
    else {
        FEATSTD_DELETE(m_binContainer[idx]);
        // Index is valid.
        static_cast<void>(m_binContainer.Remove(idx));
        return true;
    }
}

void RenderOrder::SetBinRank(const Char* binName, Int32 rank)
{
    const Int32 idx = FindBin(binName);
    if (idx < 0) {
        FEATSTD_LOG_ERROR("Set bin rank failed, bin not found: %s.", binName);
        return;
    }
    else {
        m_binContainer[idx]->SetRank(rank);
        bool (* const pComparator)(const RenderOrderBin* lhs, const RenderOrderBin* rhs) = &IsRankLess;
        m_binContainer.Sort(pComparator);
    }
}

Int32 RenderOrder::GetBinRank(const Char* binName) const
{
    const Int32 idx = FindBin(binName);
    if (idx < 0) {
        FEATSTD_LOG_ERROR("Get bin rank failed, bin not found: %s.", binName);
        return 0;
    }
    else {
        return m_binContainer[idx]->GetRank();
    }
}

void RenderOrder::SetBinSortingEnabled(const Char* binName, bool enable)
{
    const Int32 idx = FindBin(binName);
    if (idx < 0) {
        FEATSTD_LOG_ERROR("Set bin sorting enabled failed, bin not found: %s", binName);
        return;
    }
    else {
        m_binContainer[idx]->SetSortingEnabled(enable);
    }
}

bool RenderOrder::IsBinSortingEnabled(const Char* binName) const
{
    const Int32 idx = FindBin(binName);
    if (idx < 0) {
        FEATSTD_LOG_ERROR("Is bin sorting enabled failed, bin not found: %s", binName);
        return false;
    }
    else {
        return m_binContainer[idx]->IsSortingEnabled();
    }
}

void RenderOrder::SetBinOrderCriterion(const Char* binName, const OrderCriterion* orderCriterion)
{
    const Int32 idx = FindBin(binName);
    if (idx < 0) {
        FEATSTD_LOG_ERROR("Set bin order criterion failed, bin not found: %s", binName);
        return;
    }
    else {
        m_binContainer[idx]->SetOrderCriterion(orderCriterion);
    }
}

const OrderCriterion* RenderOrder::GetBinOrderCriterion(const Char* binName) const
{
    const Int32 idx = FindBin(binName);
    if (idx < 0) {
        FEATSTD_LOG_ERROR("Get bin order criterion failed, bin not found: %s", binName);
        return 0;
    }
    else {
        return m_binContainer[idx]->GetOrderCriterion();
    }
}

void RenderOrder::IterationBegin()
{
    m_nodeIndex = 0;
    m_binIndex = 0;
    //Move to next RenderOrderBin that is not empty.
    const Int32 binContainerSize = static_cast<Int32>(m_binContainer.Size());
    while ((m_binIndex < binContainerSize) && m_binContainer[m_binIndex]->IsEmpty()) {
        ++m_binIndex;
    }
}

/**
 * Returns true if there are further nodes available. This condition is false when the last
 * bin has been processed and the bin index goes out of range.
 * Remark: The node index is not relevant for this check.
 */
bool RenderOrder::IterationHasMoreNodes()
{
    return m_binIndex < static_cast<Int32>(m_binContainer.Size());
}

void RenderOrder::IterationMoveToNextNode()
{
    if (m_nodeIndex < (static_cast<Int32>(m_binContainer[m_binIndex]->GetSize() - 1)) ) {
        ++m_nodeIndex;
    }
    else  {
        m_nodeIndex = 0;
        ++m_binIndex;
        // Move to next RenderOrderBin that is not empty.
        const Int32 binContainerSize = static_cast<Int32>(m_binContainer.Size());
        while ((m_binIndex < binContainerSize) && m_binContainer[m_binIndex]->IsEmpty()) {
            ++m_binIndex;
        }
    }
}

Node* RenderOrder::IterationGetNode()
{
    return m_binContainer[m_binIndex]->GetNode(m_nodeIndex);
}

bool RenderOrder::AssignNodeToBin(Node* node)
{
    if ((node == 0) || (!node->IsRenderPrerequisiteFulfilled())) {
        // As node is undefined or cannot be rendered, it must not be assigned to any render order bin.
        return false;
    }

    const Char* binName = node->GetRenderOrderBinAssignment();
    // Assign to one of the default bins (transparent or opaque) if no bin name is set.
    if (binName == 0) {
        RenderMode* renderMode = 0;
        Appearance* appearance = node->m_appearance.GetPointerToSharedInstance();
        if (0 != appearance) {
            renderMode = appearance->m_renderMode.GetPointerToSharedInstance();
            if (0 != renderMode) {
                if (renderMode->IsInherited(RenderMode::BlendingEnabledBit)) {
                    renderMode = 0;
                }
            }
        }
        if (0 == renderMode) {
            const Camera* camera = RenderDevice::GetActiveCamera();
            if (0 != camera) {
                Appearance* cameraAppearance = camera->m_appearance.GetPointerToSharedInstance();
                if (0 != cameraAppearance) {
                    renderMode = cameraAppearance->m_renderMode.GetPointerToSharedInstance();
                }
            }
        }
        if (0 == renderMode) {
            renderMode = Renderer::GetDefaultRenderMode().GetPointerToSharedInstance();
        }

        FEATSTD_DEBUG_ASSERT(0 != renderMode); // At least Renderer::GetDefaultRenderMode() must always provide a RenderMode.
        if (renderMode->IsBlendingEnabled()) {
            return m_transparentBin.AddNode(node);
        }
        else {
            return m_opaqueBin.AddNode(node);
        }
    }

    const Int32 index = FindBin(binName, node->GetRenderOrderBinAssignmentHash());
    if (index < 0) {
        FEATSTD_LOG_ERROR("Find bin failed, bin not found: %s", binName);
        return false;
    }
    else {
        // Add Node to bin; sorting will be done later.
        return m_binContainer[index]->AddNode(node);
    }
}

bool RenderOrder::AssignVerifiedNodeToBin(Node* node)
{
    FEATSTD_DEBUG_ASSERT(node != 0); // node has to be verified
    FEATSTD_DEBUG_ASSERT(m_binContainer.Size() > 0);
    FEATSTD_DEBUG_ASSERT(m_binContainer[0] != 0);
    return m_binContainer[0]->AddNode(node);
}

void RenderOrder::SortBins()
{
    for (Int32 i = 0; i < static_cast<Int32>(m_binContainer.Size()); ++i) {
        m_binContainer[i]->Sort();
    }
}

void RenderOrder::Clear()
{
    for (Int32 i = 0; i < static_cast<Int32>(m_binContainer.Size()); ++i) {
        m_binContainer[i]->Clear();
    }
}

Int32 RenderOrder::InsertBin(RenderOrderBin* bin)
{
    if (bin == 0) {
        return -1;
    }

    const Char* binName = bin->GetName();
    FEATSTD_DEBUG_ASSERT(0 != binName);
    const UInt32 binNameHash = FeatStd::Hash::CalcHash(binName);
    bin->SetNameHash(binNameHash);

    // Find position to insert RenderOrderBin into bin container according to its rank.
    Int32 index = 0;
    bool hasNameHashCollision = false;
    for (SizeType i = 0; i < m_binContainer.Size(); ++i) {
        if (bin->GetRank() > m_binContainer[i]->GetRank()) {
            ++index;
        }

        if (binNameHash == m_binContainer[i]->GetNameHash()) {
            hasNameHashCollision = true;
            FEATSTD_LOG_ERROR("Hash of RenderOrder bin '%s' collides with '%s'. Pick another name for the bin.", bin->GetName(),
                m_binContainer[i]->GetName());
        }
    }

    m_hasNameHashCollisions = hasNameHashCollision;

    // Insert RenderOrderBin at position found.
    const bool insertOK = m_binContainer.Insert(index, bin);
    if (!insertOK) {
        FEATSTD_LOG_ERROR("Insert bin failed for %s.", bin->GetName());
        return -1;
    }

    return index;
}

Int32 RenderOrder::FindBin(const Char* binName) const
{
    if ((OpaqueBinName == binName) || (TransparentBinName == binName)) {
        for (Int32 i = 0; i < static_cast<Int32>(m_binContainer.Size()); ++i) {
            if (binName == m_binContainer[i]->GetName()) {
                return i;
            }
        }
    }
    else {
        FEATSTD_DEBUG_ASSERT(0 != binName);
        UInt32 binNameHash = FeatStd::Hash::CalcHash(binName);
        for (Int32 i = 0; i < static_cast<Int32>(m_binContainer.Size()); ++i) {
            const RenderOrderBin* renderOrderBin = m_binContainer[i];
            if (renderOrderBin->GetNameHash() == binNameHash) {
                if (StringPlatform::CompareStrings(binName, renderOrderBin->GetName()) == 0) {
                    return i;
                }
            }
        }
    }
    // bin with name given has not been found.
    return -1;
}

Int32 RenderOrder::FindBin(const Char* binName, UInt32 binNameHash) const
{
    if (m_hasNameHashCollisions) {
        return FindBin(binName);
    }

    for (Int32 i = 0; i < static_cast<Int32>(m_binContainer.Size()); ++i) {
        const RenderOrderBin* renderOrderBin = m_binContainer[i];
        if (renderOrderBin->GetNameHash() == binNameHash) {
            return i;
        }
    }

    // bin with name given has not been found.
    return -1;
}

bool RenderOrder::ResizeBinContainer(SizeType size)
{
    if (size != m_binContainer.GetCapacity()) {
        return m_binContainer.Reserve(size);
    }

    return true;
}
} // namespace Candera
