//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef Candera_RankOrderCriterion_h
#define Candera_RankOrderCriterion_h

#include <Candera/Engine3D/RenderOrder/RenderOrderCriterion.h>

namespace Candera {

/** @addtogroup RenderOrder3D
 *  @{
 */
 
// forward declaration.
class Node; 

/**
 *  @brief  class RankOrderCriterion compares two Node's render order ranks, whereas lower render order rank 
 *          is before higher render order rank.
 */
class RankOrderCriterion : public RenderOrderCriterion
{
    FEATSTD_TYPEDEF_BASE(RenderOrderCriterion);

    public:
        /**
         *  Constructor
         */
        RankOrderCriterion() {}

        /**
         *  Destructor
         */
        virtual ~RankOrderCriterion() override {}

        /**
         *  Compares two Node's render order ranks.
         *  @param a First Node whose rank is compared to Node b.
         *  @param b Second Node whose rank is compared to Node a.
         *  @return true if first Node's render order rank is lower than second Node's render order rank. 
         */
        virtual bool IsBefore(const Node& a, const Node& b) const override;

        /**
         *  Calculates the order criterion values (i.e. sorting keys) from the rank for the given nodes.
         *  @param nodeContainer     The container of nodes to compute the sorting keys for.
         *  @param sortingContainer  The container to store the sorting keys and node indices.
         *  @param nodeContainerSize The number of nodes in use in the nodeContainer. This can be
         *                           smaller than the actual size of the container
         *  @return True if all order criterion values could be prepared, False otherwise.
         */
        virtual bool PrepareRenderOrderCriterionValues(Internal::RenderOrderBin::NodeContainer& nodeContainer,
            Internal::RenderOrderBin::SortingContainer& sortingContainer, const SizeType nodeContainerSize) const override;

        /**
         *  Sort the container of the RenderOrderBin
         *  @param sortingContainer  The container to be sorted.
         *  @param containerSize     The size of the container to be sorted. This can be smaller than
         *                           the actual size of the container to sort a subsection.
         */
        virtual void Sort(Internal::RenderOrderBin::SortingContainer& sortingContainer, const SizeType containerSize) const override;

        /**
         *  Determine if the render order has changed from the previous pass to the current one.
         *  If the order has not changed, the render order bin will use the result of the previous pass,
         *  instead of sorting the current pass.
         *  @param currentBin   The sorting bin of the previous pass.
         *  @param previousBin  The sorting bin of the current pass.
         *  @return True, if the bins are different or the order has changed. False, otherwise.
         */
        virtual bool HasOrderChanged(const Internal::RenderOrderBin::SortingBin& currentBin,
            const Internal::RenderOrderBin::SortingBin& previousBin) const;
  
        FEATSTD_RTTI_DECLARATION();
};

/**
 *  @brief  class ReverseRankOrderCriterion compares two Node's render order ranks, whereas higher 
 *          render order rank is before lower render order rank.
 */
class ReverseRankOrderCriterion : public RankOrderCriterion
{
    FEATSTD_TYPEDEF_BASE(RankOrderCriterion);

    public:
        /**
         *  Constructor
         */
        ReverseRankOrderCriterion() {}

        /**
         *  Destructor
         */
        virtual ~ReverseRankOrderCriterion() override {}

        /**
         *  Compares two Node's render order ranks.
         *  @param a First Node whose rank is compared to Node b.
         *  @param b Second Node whose rank is compared to Node a.
         *  @return true if first Node's render order rank is higher than second Node's render order rank. 
         */
        virtual bool IsBefore(const Node& a, const Node& b) const override;

        /**
         *  Sort the container of the RenderOrderBin
         *  @param sortingContainer  The container to be sorted.
         *  @param containerSize     The size of the container to be sorted. This can be smaller than
         *                           the actual size of the container to sort a subsection.
         */
        virtual void Sort(Internal::RenderOrderBin::SortingContainer& sortingContainer, const SizeType containerSize) const override;

        /**
         *  Determine if the render order has changed from the previous pass to the current one.
         *  If the order has not changed, the render order bin will use the result of the previous pass,
         *  instead of sorting the current pass.
         *  @param currentBin   The sorting bin of the previous pass.
         *  @param previousBin  The sorting bin of the current pass.
         *  @return True, if the bins are different or the order has changed. False, otherwise.
         */
        virtual bool HasOrderChanged(const Internal::RenderOrderBin::SortingBin& currentBin,
            const Internal::RenderOrderBin::SortingBin& previousBin) const;
        
        FEATSTD_RTTI_DECLARATION();
};

 /** @} */ // end of RenderOrder3D
 
}

#endif // Candera_RankOrderCriterion_h
