//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "RankOrderCriterion.h"
#include <Candera/Engine3D/Core/Node.h>

namespace Candera {

using namespace Diagnostics;
using namespace Internal;

FEATSTD_LOG_SET_REALM(LogRealm::CanderaEngine3D);

/**
 *  class RenderOrderRank
 */

FEATSTD_RTTI_DEFINITION(RankOrderCriterion, RenderOrderCriterion)

bool RankOrderCriterion::IsBefore(const Node& a, const Node& b) const
{
    return a.GetRenderOrderRank() < b.GetRenderOrderRank();
}

bool RankOrderCriterion::PrepareRenderOrderCriterionValues(RenderOrderBin::NodeContainer& nodeContainer,
    RenderOrderBin::SortingContainer& sortingContainer, const SizeType nodeContainerSize) const
{
    if (!Reserve<UInt32>(sortingContainer, nodeContainerSize, 0)) {
        FEATSTD_LOG_ERROR("Prepare render order criterion value failed, could not reserve space in bins.");
        return false;
    }

    UInt16 indexOfNode = 0;
    for (Int i = 0; i < static_cast<Int>(nodeContainerSize); ++i) {
        Node& n = *(nodeContainer[i]);

        sortingContainer[indexOfNode] = RenderOrderBin::SortingElement(static_cast<UInt32>(n.GetRenderOrderRank()), indexOfNode);
        indexOfNode++;
    }

    return true;
}

struct RankOrderComparator
{
    bool operator()(const RenderOrderBin::SortingElement& a, const RenderOrderBin::SortingElement& b) const {
        return a.IsBeforeInt32(b);
    }
};

void RankOrderCriterion::Sort(RenderOrderBin::SortingContainer& sortingContainer, const SizeType containerSize) const
{
    RankOrderComparator comparator;
    sortingContainer.Sort(comparator, 0, containerSize - 1);
}

bool RankOrderCriterion::HasOrderChanged(const RenderOrderBin::SortingBin& currentBin,
    const RenderOrderBin::SortingBin& previousBin) const
{
    RankOrderComparator comparator;
    RenderOrderBin::SortingElement startingKey(static_cast<UInt32>(FeatStd::Internal::NativeTypeLimit<Int32>::Min()), 0);
    return RenderOrderCriterion::HasOrderChanged(currentBin, previousBin, comparator, startingKey);
}

/**
 * class ReverseRenderOrderRank
 */

FEATSTD_RTTI_DEFINITION(ReverseRankOrderCriterion, RankOrderCriterion)

bool ReverseRankOrderCriterion::IsBefore(const Node& a, const Node& b) const
{
    return a.GetRenderOrderRank() > b.GetRenderOrderRank();
}

struct ReverseRankOrderComparator
{
    bool operator()(const RenderOrderBin::SortingElement& a, const RenderOrderBin::SortingElement& b) const {
        return b.IsBeforeInt32(a);
    }
};

void ReverseRankOrderCriterion::Sort(RenderOrderBin::SortingContainer& sortingContainer, const SizeType containerSize) const
{
    ReverseRankOrderComparator comparator;
    sortingContainer.Sort(comparator, 0, containerSize - 1);
}

bool ReverseRankOrderCriterion::HasOrderChanged(const RenderOrderBin::SortingBin& currentBin,
    const RenderOrderBin::SortingBin& previousBin) const
{
    ReverseRankOrderComparator comparator;
    RenderOrderBin::SortingElement startingKey(static_cast<UInt32>(FeatStd::Internal::NativeTypeLimit<Int32>::Max()), 0);
    return RenderOrderCriterion::HasOrderChanged(currentBin, previousBin, comparator, startingKey);
}

}
