//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "DistanceToCameraOrderCriterion.h"
#include <Candera/Engine3D/Core/Node.h>
#include <Candera/Engine3D/Core/Camera.h>
#include <Candera/System/Mathematics/Vector3.h>
#include <Candera/System/Diagnostics/Log.h>
#include <CanderaPlatform/Device/Common/Base/RenderDevice.h>

namespace Candera {
using namespace Diagnostics;
using namespace Internal;
    FEATSTD_LOG_SET_REALM(LogRealm::CanderaEngine3D);

    FEATSTD_RTTI_DEFINITION(CameraOrderCriterion, RenderOrderCriterion)

void CameraOrderCriterion::PrepareRenderOrderCriterionValue(Candera::Node &n) const
{
    const Camera* camera = RenderDevice::GetActiveCamera();
    if (camera == 0) {
        FEATSTD_LOG_ERROR("Prepare render order criterion value failed, no active camera set.");
        return;
    }

    Float distanceToNode = Math::MaxFloat();

    switch (m_distFunc) {
        case CameraPositionToBoundingSphere: {
            // Calculate distance from camera's world position to node's bounding sphere in world space.
            distanceToNode = camera->GetWorldPosition().GetDistanceTo(n.GetWorldCenter()) - n.GetWorldRadius();
            break;
        }

        case CameraPositionToBoundingBox: {
            Vector3 wobbBounds[8];
            n.GetWorldOrientedBoundingBox(wobbBounds);
            const Matrix4& viewMatrix = camera->GetViewMatrix();
            // Transform bounding box into camera space, where length of bounding box coordinates is equal to distance to camera.
            for (Int i = 0; i < 8; i++) {
                wobbBounds[i].TransformCoordinate(viewMatrix);
                Math::SetToMax(wobbBounds[i].GetSquaredLength(), distanceToNode);
            }
        break;
        }

        case ViewPlaneToBoundingSphere: {
            const Vector3 cameraWorldLookAtVector = camera->GetWorldLookAtVector();
            // DotProduct represents plane equation [a*x + b*y + c*z + distance = 0] without distance.
            // Note: Only the viewing planes orientation is considered. If the camera moves along the look at vector, the order remains.
            distanceToNode = cameraWorldLookAtVector.GetDotProduct( n.GetWorldCenter() - (cameraWorldLookAtVector * n.GetWorldRadius()));
            break;
        }

        case ViewPlaneToBoundingBox: {
            const Vector3 cameraWorldLookAtVector = camera->GetWorldLookAtVector();
            Vector3 wobbBounds[8];
            n.GetWorldOrientedBoundingBox(wobbBounds);
            // DotProduct represents plane equation [a*x + b*y + c*z + distance = 0] without distance.
            // Note: Only the viewing planes orientation is considered. If the camera moves along the look at vector, the order remains.
            for (Int i = 0; i < 8; i++) {
                Math::SetToMax(cameraWorldLookAtVector.GetDotProduct(wobbBounds[i]), distanceToNode);
            }
            break;
        }

        default:
            break;
    }

    SetRenderOrderCriterionValue(n, OrderCriterionValue(distanceToNode));
}

bool CameraOrderCriterion::PrepareRenderOrderCriterionValues(RenderOrderBin::NodeContainer& nodeContainer,
    RenderOrderBin::SortingContainer& sortingContainer, const SizeType nodeContainerSize) const
{
    const Camera* camera = RenderDevice::GetActiveCamera();
    if (camera == 0) {
        FEATSTD_LOG_ERROR("Prepare render order criterion value failed, no active camera set.");
        return false;
    }

    if (!Reserve<Float>(sortingContainer, nodeContainerSize, 0.0F)) {
        FEATSTD_LOG_ERROR("Prepare render order criterion value failed, could not reserve space in bins.");
        return false;
    }

    UInt16 indexOfNode = 0;

    switch (m_distFunc) {
    case CameraPositionToBoundingSphere: {
        const Vector3& cameraWorldPosition = camera->GetWorldPosition();
        for (UInt i = 0; i < static_cast<UInt>(nodeContainerSize); ++i) {
            Node& n = *(nodeContainer[i]);

            // Calculate distance from camera's world position to node's bounding sphere in world space.
            Float distanceToNode = cameraWorldPosition.GetDistanceTo(n.GetWorldCenter()) - n.GetWorldRadius();

            sortingContainer[indexOfNode] = RenderOrderBin::SortingElement(distanceToNode, indexOfNode);
            indexOfNode++;
        }
        break;
    }

    case CameraPositionToBoundingBox: {
        for (UInt i = 0; i < static_cast<UInt>(nodeContainerSize); ++i) {
            Node& n = *(nodeContainer[i]);
            Float distanceToNode = Math::MaxFloat();

            const Matrix4& viewMatrix = camera->GetViewMatrix();
            Vector3 wobbBounds[8];
            n.GetWorldOrientedBoundingBox(wobbBounds);
            // Transform bounding box into camera space, where length of bounding box coordinates is equal to distance to camera.
            for (Int j = 0; j < 8; ++j) {
                wobbBounds[j].TransformCoordinate(viewMatrix);
                Math::SetToMax(wobbBounds[j].GetSquaredLength(), distanceToNode);
            }

            sortingContainer[indexOfNode] = RenderOrderBin::SortingElement(distanceToNode, indexOfNode);
            indexOfNode++;
        }

        break;
    }

    case ViewPlaneToBoundingSphere: {
        const Vector3& cameraWorldLookAtVector = camera->GetWorldLookAtVector();
        for (UInt i = 0; i < static_cast<UInt>(nodeContainerSize); ++i) {
            Node& n = *(nodeContainer[i]);

            // DotProduct represents plane equation [a*x + b*y + c*z + distance = 0] without distance.
            // Note: Only the viewing planes orientation is considered. If the camera moves along the look at vector, the order remains.
             Float distanceToNode = cameraWorldLookAtVector.GetDotProduct(n.GetWorldCenter() - (cameraWorldLookAtVector * n.GetWorldRadius()));

             sortingContainer[indexOfNode] = RenderOrderBin::SortingElement(distanceToNode, indexOfNode);
            indexOfNode++;
        }

        break;
    }

    case ViewPlaneToBoundingBox: {
        const Vector3& cameraWorldLookAtVector = camera->GetWorldLookAtVector();
        for (UInt i = 0; i < static_cast<UInt>(nodeContainerSize); ++i) {
            Node& n = *(nodeContainer[i]);
            Float distanceToNode = Math::MaxFloat();

            Vector3 wobbBounds[8];
            n.GetWorldOrientedBoundingBox(wobbBounds);
            // DotProduct represents plane equation [a*x + b*y + c*z + distance = 0] without distance.
            // Note: Only the viewing planes orientation is considered. If the camera moves along the look at vector, the order remains.
            for (Int j = 0; j < 8; ++j) {
                Math::SetToMax(cameraWorldLookAtVector.GetDotProduct(wobbBounds[j]), distanceToNode);
            }

            sortingContainer[indexOfNode] = RenderOrderBin::SortingElement(distanceToNode, indexOfNode);
            indexOfNode++;
        }

        break;
    }

    default:
        FEATSTD_DEBUG_FAIL();
        return false;
    }

    return true;
}

/**
 * DistanceToCameraOrderCriterion
 */

    FEATSTD_RTTI_DEFINITION(DistanceToCameraOrderCriterion, CameraOrderCriterion)

bool DistanceToCameraOrderCriterion::IsBefore(const Node& a, const Node& b) const
{
    return GetRenderOrderCriterionValue(a).GetFloat() < GetRenderOrderCriterionValue(b).GetFloat();
}

struct DistanceToCameraComparator
{
    bool operator()(const RenderOrderBin::SortingElement& a, const RenderOrderBin::SortingElement& b) const {
        return a.IsBeforeFloat(b);
    }
};

void DistanceToCameraOrderCriterion::Sort(RenderOrderBin::SortingContainer& sortingContainer, const SizeType containerSize) const
{
    DistanceToCameraComparator comparator;
    sortingContainer.Sort(comparator, 0, containerSize - 1);
}

bool DistanceToCameraOrderCriterion::HasOrderChanged(const RenderOrderBin::SortingBin& currentBin,
    const RenderOrderBin::SortingBin& previousBin) const
{
    DistanceToCameraComparator comparator;
    RenderOrderBin::SortingElement startingKey(Math::MinFloat(), 0);
    return RenderOrderCriterion::HasOrderChanged(currentBin, previousBin, comparator, startingKey);
}

/**
 * ReverseDistanceToCameraOrderCriterion
 */

FEATSTD_RTTI_DEFINITION(ReverseDistanceToCameraOrderCriterion, CameraOrderCriterion)

bool ReverseDistanceToCameraOrderCriterion::IsBefore(const Node& a, const Node& b) const
{
    return GetRenderOrderCriterionValue(b).GetFloat() < GetRenderOrderCriterionValue(a).GetFloat();
}

struct ReverseDistanceToCameraComparator
{
    bool operator()(const RenderOrderBin::SortingElement& a, const RenderOrderBin::SortingElement& b) const {
        return b.IsBeforeFloat(a);
    }
};

void ReverseDistanceToCameraOrderCriterion::Sort(RenderOrderBin::SortingContainer& sortingContainer, const SizeType containerSize) const
{
    ReverseDistanceToCameraComparator comparator;
    sortingContainer.Sort(comparator, 0, containerSize - 1);
}

bool ReverseDistanceToCameraOrderCriterion::HasOrderChanged(const RenderOrderBin::SortingBin& currentBin,
    const RenderOrderBin::SortingBin& previousBin) const
{
    ReverseDistanceToCameraComparator comparator;
    RenderOrderBin::SortingElement startingKey(Math::MaxFloat(), 0);
    return RenderOrderCriterion::HasOrderChanged(currentBin, previousBin, comparator, startingKey);
}

}
