//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "BatchOrderCriterion.h"
#include <Candera/Engine3D/Core/Camera.h>
#include <Candera/Engine3D/Core/PerspectiveProjection.h>
#include <CanderaPlatform/Device/Common/Base/RenderDevice.h>

namespace Candera {
using namespace Diagnostics;
using namespace Internal;

FEATSTD_RTTI_DEFINITION(BatchOrderCriterion, RenderOrderCriterion)
FEATSTD_LOG_SET_REALM(LogRealm::CanderaEngine3D);

bool BatchOrderCriterion::PrepareRenderOrderCriterionValues(RenderOrderBin::NodeContainer& nodeContainer,
RenderOrderBin::SortingContainer& sortingContainer, const SizeType nodeContainerSize) const
{
    const Camera* camera = RenderDevice::GetActiveCamera();
    if (0 == camera) {
        FEATSTD_LOG_ERROR("Prepare render order criterion value failed, no active camera set.");
        return false;
    }

    const Float maxUInt16 = static_cast<Float>(FeatStd::Internal::NativeTypeLimit<UInt16>::Max());
    Float offsetZ = 0.0F;
    Float scaleToUInt16 = 1.0F;
    Projection* projection = camera->m_projection.GetPointerToSharedInstance();
    if (0 != projection) {
        PerspectiveProjection* perspectiveProjection = Dynamic_Cast<PerspectiveProjection*>(projection);
        if (0 != perspectiveProjection) {
            offsetZ = perspectiveProjection->GetFarZ();
            scaleToUInt16 = maxUInt16 / (perspectiveProjection->GetFarZ() * 2.0F);
        }
    }

    if (!Reserve<UInt32>(sortingContainer, nodeContainerSize, 0)) {
        FEATSTD_LOG_ERROR("Prepare render order criterion value failed, could not reserve space in bins.");
        return false;
    }

    UInt16 indexOfNode = 0;
    const Vector3& cameraWorldPosition = camera->GetWorldPosition();
    const Vector3& cameraForward = camera->GetWorldLookAtVector();

    for (UInt i = 0; i < static_cast<UInt>(nodeContainerSize); ++i) {
        Node& node = *(nodeContainer[i]);

        Vector3 cameraToNode = node.GetWorldCenter() - cameraWorldPosition;
        Float distanceToNode = cameraForward.GetDotProduct(cameraToNode) + offsetZ;
        distanceToNode -= node.GetWorldRadius();
        distanceToNode *= scaleToUInt16;

        if (distanceToNode < 0.0F) {
            distanceToNode = 0.0F;
        }
        else {
            if (distanceToNode > maxUInt16) {
                distanceToNode = maxUInt16;
            }
        }
        
        UInt16 secondaryKey = static_cast<UInt16>(distanceToNode);
        UInt32 key = 0;
        UInt32 batchableKey = 0;
        UInt32 vertexBufferKey = 0;

        Renderable* renderable = Dynamic_Cast<Renderable*>(&node);
        if (0 != renderable) {
            if (renderable->m_isBatchable) {
                batchableKey = 1;
            }

            VertexBuffer* vertexBuffer = renderable->m_vertexBuffer.GetPointerToSharedInstance();
            // Has been verified by Renderable::IsRenderPrerequisiteFulfilled() via RenderOrder::AssignNodeToBin() before.
            // The assert is just here to catch code changes.
            FEATSTD_DEBUG_ASSERT(0 != vertexBuffer);

            vertexBufferKey = vertexBuffer->GetInstanceId();
        }

        Candera::Appearance* appearance = node.m_appearance.GetPointerToSharedInstance();

        if (m_mode == BatchOrderCriterion::Appearance) {
            UInt32 appearanceKey = 0;
            if (0 != appearance) {
                appearanceKey = appearance->GetInstanceId();
            }

            enum KeyBits {
                BATCHABLE = 1,
                VERTEXBUFFER = 16,  // Can handle up to 2^VERTEXBUFFER VertexBuffer objects at runtime.
                APPEARANCE = 15     // Can handle up to 2^APPEARANCE Appearance objects at runtime.
            };

            // assert that key bits fit into 32 bits
            FEATSTD_COMPILETIME_ASSERT((BATCHABLE + VERTEXBUFFER + APPEARANCE) <= (sizeof(key) * 8));

            // Assert that there are not more runtime objects (i.e. higher Ids) than the key can hold.
            FEATSTD_DEBUG_ASSERT(batchableKey < (1 << BATCHABLE));
            FEATSTD_DEBUG_ASSERT(vertexBufferKey < (1 << VERTEXBUFFER));
            FEATSTD_DEBUG_ASSERT(appearanceKey < (1 << APPEARANCE));

            batchableKey = batchableKey << (VERTEXBUFFER + APPEARANCE);
            vertexBufferKey = vertexBufferKey << (APPEARANCE);
            key = batchableKey | vertexBufferKey | appearanceKey;
        }
        else {
            UInt32 shaderKey = 0;
            UInt32 renderModeKey = 0;

            if (0 != appearance) {
                Shader* shader = appearance->m_shader.GetPointerToSharedInstance();
                // Has been verified by Node::IsRenderPrerequisiteFulfilled() via RenderOrder::AssignNodeToBin() before.
                // The assert is just here to catch code changes.
                FEATSTD_DEBUG_ASSERT(0 != shader);
                shaderKey = shader->GetInstanceId();

                RenderMode* renderMode = appearance->m_renderMode.GetPointerToSharedInstance();
                if (0 != renderMode) {
                    renderModeKey = renderMode->GetInstanceId();
                }
            }

            enum KeyBits {
                BATCHABLE = 1,
                VERTEXBUFFER = 12,
                SHADER = 10,
                RENDERMODE = 9
            };

            FEATSTD_COMPILETIME_ASSERT((BATCHABLE + VERTEXBUFFER + SHADER + RENDERMODE) <= (sizeof(key) * 8));

            // assert that there are not more runtime objects (i.e. higher Ids) than the key can hold.
            FEATSTD_DEBUG_ASSERT(batchableKey < (1 << BATCHABLE));
            FEATSTD_DEBUG_ASSERT(vertexBufferKey < (1 << VERTEXBUFFER));
            FEATSTD_DEBUG_ASSERT(shaderKey < (1 << SHADER));
            FEATSTD_DEBUG_ASSERT(renderModeKey < (1 << RENDERMODE));

            batchableKey = batchableKey << (VERTEXBUFFER + SHADER + RENDERMODE);
            vertexBufferKey = vertexBufferKey << (SHADER + RENDERMODE);
            shaderKey = shaderKey << (RENDERMODE);
            key = batchableKey | vertexBufferKey | shaderKey | renderModeKey;
        }

        sortingContainer[indexOfNode] = RenderOrderBin::SortingElement(key, secondaryKey, indexOfNode);
        indexOfNode++;
    }

    return true;
}

struct BatchOrderComparator
{
    bool operator()(const RenderOrderBin::SortingElement& a, const RenderOrderBin::SortingElement& b) const {
        if (a.GetUInt32() == b.GetUInt32()) {
            return (a.GetSecondaryKey() < b.GetSecondaryKey());
        }
        return a.IsBeforeUInt32(b);
    }
};

void BatchOrderCriterion::Sort(RenderOrderBin::SortingContainer& sortingContainer, const SizeType containerSize) const
{
    BatchOrderComparator comparator;
    sortingContainer.Sort(comparator, 0, containerSize - 1);
}

bool BatchOrderCriterion::HasOrderChanged(const RenderOrderBin::SortingBin& currentBin,
    const RenderOrderBin::SortingBin& previousBin) const
{
    BatchOrderComparator comparator;
    RenderOrderBin::SortingElement startingKey(FeatStd::Internal::NativeTypeLimit<UInt32>::Min(),
        FeatStd::Internal::NativeTypeLimit<UInt16>::Min(), 0);
    return RenderOrderCriterion::HasOrderChanged(currentBin, previousBin, comparator, startingKey);
}

}
