//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include <Candera/Engine3D/Layout/CanvasTextLayouter.h>
#include <Candera/Engine3D/Canvas/CanvasText.h>
#include <Candera/TextEngine/TextSize.h>

using namespace Candera::TextRendering;

namespace Candera {

    static TextSize Vector2ToTextSize(const Vector2& size)
    {
        return TextSize(static_cast<Int16>(Math::Minimum(size.GetX(), static_cast<Float>(FeatStd::Internal::Limits<Int16>::Max()))),
                        static_cast<Int16>(Math::Minimum(size.GetY(), static_cast<Float>(FeatStd::Internal::Limits<Int16>::Max()))));
    }

    Candera::Vector2 CanvasTextLayouter::OnMeasure(const AbstractNodePointer& node, const Vector2& clientArea)
    {
        if (node.IsValid()) {
            CanvasText* textNode = Dynamic_Cast<CanvasText*>(node.ToNode());
            if (textNode == 0) {
                return Vector2(-1.0F, -1.0F);
            }

            Vector2 changedClientAreaSize = clientArea;
            /*
            const Vector2& setSize = Layouter::GetSize(*node.ToCanderaObject());
            if ((setSize.GetX() >= 0) && (setSize.GetX() < changedClientAreaSize.GetX())) {
                changedClientAreaSize.SetX(setSize.GetX());
            }
            if ((setSize.GetY() >= 0) && (setSize.GetY() < changedClientAreaSize.GetY())) {
                changedClientAreaSize.SetY(setSize.GetY());
            }*/
            HorizontalAlignment hAlignment = GetHorizontalAlignment(*node.ToCanderaObject());
            VerticalAlignment vAlignment = GetVerticalAlignment(*node.ToCanderaObject());
            bool isRightToLeft = GetLanguageSensitiveDirection(node) == LayoutAlignment::LayoutDirection::RightToLeftDirection;

            textNode->SetLayoutRestriction(Vector2ToTextSize(changedClientAreaSize));
            textNode->ResolveAutoHorizontalAlignment(GetLanguageSensitiveTextAlignment(*textNode));
            textNode->SetRightToLeftLayoutDirection(isRightToLeft);
            Candera::Rectangle resultBoundingRectangle;
            textNode->GetComputedBoundingRectangle(resultBoundingRectangle);
            Vector2 resultVector(resultBoundingRectangle.GetWidth(), resultBoundingRectangle.GetHeight());
            TextSize innerSize = textNode->GetSize();
            if (innerSize.GetWidth() > 0) {
                resultVector.SetX(static_cast<FeatStd::Float>(innerSize.GetWidth()));
            }
            if (innerSize.GetHeight() > 0) {
                resultVector.SetY(static_cast<FeatStd::Float>(innerSize.GetHeight()));
            }
            Vector2 offset(0.0F, 0.0F);

            bool isClientXSet = (changedClientAreaSize.GetX() >= 0.0F) && (changedClientAreaSize.GetX() < static_cast<Float>(FeatStd::Internal::Limits<Int16>::Max()));
            bool isClientYSet = (changedClientAreaSize.GetY() >= 0.0F) && (changedClientAreaSize.GetY() < static_cast<Float>(FeatStd::Internal::Limits<Int16>::Max()));

            if (isClientXSet) {
                Float sizeDifference = resultVector.GetX() - changedClientAreaSize.GetX();
                if (sizeDifference > 0.0F) {
                    switch (hAlignment) {
                    case Candera::HLeft:
                        if (isRightToLeft) {
                            offset.SetX(offset.GetX() - sizeDifference);
                        }
                        break;
                    case Candera::HCenter:
                        offset.SetX(offset.GetX() - sizeDifference/2.0F);
                        break;
                    case Candera::HRight:
                        if (!isRightToLeft) {
                            offset.SetX(offset.GetX() - sizeDifference);
                        }
                        break;
                    case Candera::HStretch:
                        break;
                    default:
                        break;
                    }
                    resultVector.SetX(changedClientAreaSize.GetX());
                }

            }
            if (isClientYSet) {
                Float sizeDifference = resultVector.GetY() - changedClientAreaSize.GetY();
                if (sizeDifference > 0.0F) {
                    switch (vAlignment) {
                    case Candera::VTop:
                        break;
                    case Candera::VCenter:
                        offset.SetY(offset.GetY() - sizeDifference / 2.0F);
                        break;
                    case Candera::VBottom:
                            offset.SetY(offset.GetY() - sizeDifference);
                        break;
                    case Candera::HStretch:
                        break;
                    default:
                        break;
                    }
                    resultVector.SetY(changedClientAreaSize.GetY());
                }
            }
            textNode->SetResultOffset(static_cast<Int16>(offset.GetX()), static_cast<Int16>(offset.GetY()));

            return resultVector;
        }
        return Vector2();
    }


    void CanvasTextLayouter::OnArrange(const AbstractNodePointer& node, const Candera::Rectangle& clientArea)
    {
        if (node.IsValid()) {
            CanvasText* textNode = Dynamic_Cast<CanvasText*>(node.ToNode());
            if (textNode == 0) {
                SetNodePosition(node, clientArea.GetPosition());
            }
            else {
                Candera::Rectangle boundingRect;
                GetParentSpaceAxisAlignedBoundingRect(node, boundingRect);

                Candera::Rectangle rect(0.0F, 0.0F, boundingRect.GetWidth(), boundingRect.GetHeight());
                AlignRectangle(clientArea, rect, node);

                SetNodePosition(node, rect.GetPosition());
            }

            Vector2 actualSize = GetPreferredSize(node);
            if (GetHorizontalAlignment(*node.ToCanderaObject()) == HStretch) {
                actualSize.SetX(clientArea.GetWidth());
            }
            SetArrangeActualSize(actualSize);
        }
    }


    Candera::Layouter* CanvasTextLayouter::Clone() const
    {
        return &GetDefault();
    }


    void CanvasTextLayouter::Dispose()
    {

    }


    Candera::CanvasTextLayouter& CanvasTextLayouter::GetDefault()
    {
        FEATSTD_SYNCED_STATIC_OBJECT(CanvasTextLayouter, s_canvasTextLayouter);
        return s_canvasTextLayouter;
    }
    static CanvasText::HorizontalTextAlignment::Enum ConvertAlignmentLayoutToText(HorizontalAlignment alignment)
    {
        CanvasText::HorizontalTextAlignment::Enum result;
        switch (alignment) {
        case Candera::HLeft:
            result = CanvasText::HorizontalTextAlignment::Left;
            break;
        case Candera::HCenter:
            result = CanvasText::HorizontalTextAlignment::Center;
            break;
        case Candera::HRight:
            result = CanvasText::HorizontalTextAlignment::Right;
            break;
        case Candera::HStretch:
            result = CanvasText::HorizontalTextAlignment::Justified;
            break;
        default:
            result = CanvasText::HorizontalTextAlignment::Left;
            // should not be reached
            break;
        }
        return result;
    }

    CanvasText::HorizontalTextAlignment::Enum CanvasTextLayouter::GetLanguageSensitiveTextAlignment(CanvasText& textNode) const
    {
        CanvasText::HorizontalTextAlignment::Enum hTextAlign = textNode.GetHorizontalTextAlignment();
        CanvasText::HorizontalTextAlignment::Enum result = hTextAlign;
        switch (hTextAlign) {
        case CanvasText::HorizontalTextAlignment::Auto:
            result = ConvertAlignmentLayoutToText(GetHorizontalAlignment(static_cast<const CanderaObject&>(textNode))); break;
        case CanvasText::HorizontalTextAlignment::Left:
            break;
        case CanvasText::HorizontalTextAlignment::Center:
            break;
        case CanvasText::HorizontalTextAlignment::Right:
            break;
        case CanvasText::HorizontalTextAlignment::Justified:
            break;
        default:
            break;
        }

        AbstractNodePointer abstractNode(const_cast<CanvasText*>(&textNode));
        if (GetLanguageSensitiveDirection(abstractNode) == LayoutAlignment::LayoutDirection::RightToLeftDirection) {
            if (result == CanvasText::HorizontalTextAlignment::Left) {
                result = CanvasText::HorizontalTextAlignment::Right;
            }
            else {
                if (result == CanvasText::HorizontalTextAlignment::Right) {
                    result = CanvasText::HorizontalTextAlignment::Left;
                }
            }
        }

        return result;
    }
}
