//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_VERTEXGEOMETRYTOOLS_H)
#define CANDERA_VERTEXGEOMETRYTOOLS_H

#include <Candera/Environment.h>
#include <Candera/System/MemoryManagement/MemoryManagement.h>
#include <Candera/Engine3D/Core/VertexGeometry.h>

namespace Candera {
    namespace Internal {
        /**  @addtogroup Core3D
         *   @{
         */

        /**
         *  @brief VertexGeometryTools is a collection of function useful
         *  for conversions between instances of VertexGeometry::VertexDataType.
         */
        class VertexGeometryTools
        {
        public:
            class BaseBufferConverter;
            
            /**
             *  StoreBuffer::DataTypes are separated into these categories.
             */
            enum BaseType
            {
                Float32Type,    ///< 32bit floating point type
                UInt8Type,      ///< 8 bit unsigned integer type
                Int8Type,       ///< 8 bit integer type
                UInt16Type,     ///< 16 bit unsigned integer type
                Int16Type,      ///< 16 bit integer type
                Float16Type,    ///< 16 bit floating point type
                UInt32Type,     ///< 32 bit unsigned integer type
                Int32Type,      ///< 32 bit integer type
                BaseTypeCount   ///< Number of BaseTypes
            };

            /**
             *  @brief Map StoreBuffer::DataTypes to useful traits.
             */
            struct DataTypeTraits
            {
                BaseType type;
                UInt8 count;
            };

            /**
             * @brief Class used for converting buffers.
             */
            class Converter
            {
                friend class VertexGeometryTools;
            public:
                Converter();
                ~Converter();

                /**
                 *  Convert a buffer from one VertexGeometry::VertexDataType to another.
                 *  @param dstBuffer Buffer to store the new data.
                 *  @param dstCount Number of channels within the destination buffer.
                 *  @param dstStride Stride between elements within the destination buffer.
                 *  @param srcBuffer Buffer from which to retrieve the data.
                 *  @param srcCount Number of channels within the source buffer.
                 *  @param srcStride Stride between elements within the source buffer.
                 *  @param itemCount number of vertices to process.
                 */
                virtual void ConvertBuffer(
                    void* dstBuffer, UInt8 dstCount, UInt16 dstStride,
                    const void* srcBuffer, UInt8 srcCount, UInt16 srcStride,
                    UInt32 itemCount) const;

                /**
                 *  Convert an item from one VertexGeometry::VertexDataType to another.
                 *  @param dstBuffer Buffer to store the new data.
                 *  @param dstCount Number of channels within the destination buffer.
                 *  @param srcBuffer Buffer from which to retrieve the data.
                 *  @param srcCount Number of channels within the source buffer.
                 */
                virtual void ConvertItem(
                    void* dstBuffer, UInt8 dstCount, 
                    const void* srcBuffer, UInt8 srcCount) const;

            private:
                Converter(BaseType dstType, BaseType srcType);
                const BaseBufferConverter* m_baseConverter;
            };

            /**
             *  Retrieves information about a data type.
             *  @param type Data type for which to retrieve the information.
             *  @return A traits object describing the datatype.
             */
            static DataTypeTraits GetTypeTraits(VertexGeometry::VertexDataType type);

            /**
             *  Retrieves an object that allows conversion from one data type to another.
             *  The converter is configured to convert from the source base type to the
             *  destination base type.
             *  @param dstType Base type of the destination buffer.
             *  @param srcType Base type of the source buffer.
             *  @return A converter that can transform object from srcType to dstType.
             */
            static Converter GetConverter(BaseType dstType, BaseType srcType);
        };

        /**  @} */ // end of Core3D
    }
} // namespace

#endif    // CANDERA_VERTEXGEOMETRYTOOLS_H
