//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_VertexGeometryModifier_H)
#define CANDERA_VertexGeometryModifier_H

#include <Candera/Environment.h>
#include <Candera/Engine3D/Core/VertexGeometryTools.h>

namespace Candera {
    class VertexGeometry;

    /**  @addtogroup Core3D
     *   @{
     */

    /**
     *  @brief  In conjunction with a VertexAccessor, the VertexGeometryModifier helps to
     *  effectively modify VertexGeometry objects.
     */
    class VertexGeometryModifier
    {
    public:
        /**
         *  Constructor
         *  @param geometry Geometry from which to retrieve vertex format.
         *  @param usage Usage of this value.
         *  @param usageIndex Usage index of this value.
         */
        VertexGeometryModifier(const VertexGeometry& geometry, VertexGeometry::VertexUsage usage, UInt8 usageIndex);

        /**
         *  Set the value of the handled vertex element to the given vertex buffer.
         *  @param vertex Pointer to the memory location where the vertex data is stored.
         *  @param x X component of this value.
         *  @param y Y component of this value.
         *  @param z Z component of this value.
         *  @param w W component of this value.
         */
        void SetVertexElement(void *vertex, Float x, Float y = 0.F, Float z = 0.F, Float w = 1.F) const;

        /**
         *  Set the value of the handled vertex element to the given vertex buffer.
         *  @param vertex Pointer to the memory location where the vertex data is stored.
         *  @param t Table containing the components.
         *  @param tSize Number of components within the table.
         */
        void SetVertexElement(void *vertex, const Float *t, UInt8 tSize) const;

        /**
         *  Set the value of the handled vertex element to the given vertex buffer.
         *  @param vertex Pointer to the memory location where the vertex data is stored.
         *  @param x X component of this value.
         *  @param y Y component of this value.
         *  @param z Z component of this value.
         *  @param w W component of this value.
         */
        void SetVertexElement(void *vertex, Int x, Int y = 0, Int z = 0, Int w = 0) const;

        /**
         *  Set the value of the handled vertex element to the given vertex buffer.
         *  @param vertex Pointer to the memory location where the vertex data is stored.
         *  @param t Table containing the components.
         *  @param tSize Number of components within the table.
         */
        void SetVertexElement(void *vertex, const Int *t, UInt8 tSize) const;

        /**
         *  Retrieve the value of the handled vertex element from the given vertex buffer.
         *  @param vertex Pointer to the memory location where the vertex data is stored.
         *  @param t Table containing the components.
         *  @param tSize Number of components within the table.
         */
        void GetVertexElement(const void *vertex, Float *t, UInt8 tSize) const;

        /**
         *  Retrieve the value of the handled vertex element from the given vertex buffer.
         *  @param vertex Pointer to the memory location where the vertex data is stored.
         *  @param t Table containing the components.
         *  @param tSize Number of components within the table.
         */
        void GetVertexElement(const void *vertex, Int *t, UInt8 tSize) const;

    private:
        typedef Candera::Internal::VertexGeometryTools VertexGeometryTools;
        typedef VertexGeometryTools::Converter Converter;

        UInt8 m_count;
        UInt16 m_offset;
        Converter m_floatConverter;
        Converter m_intConverter;
        Converter m_floatConverterInverse;
        Converter m_intConverterInverse;
    };

    /**  @} */ // end of Core3D
} // namespace Candera

#endif    // CANDERA_VertexGeometryModifier_H
