//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_VERTEXGEOMETRYACCESSORS_H)
#define CANDERA_VERTEXGEOMETRYACCESSORS_H

#include <Candera/Environment.h>
#include <Candera/Engine3D/Core/VertexGeometry.h>

namespace Candera {

/**  @addtogroup Core3D
 *   @{
 */

    /**
     * Safely access raw vertices in a VertexGeometry. 
     */
    class VertexAccessor {
        public:
            /**
             * Mutable VertexAccessor constructor.
             * @param geometry VertexGeometry that contains the vertices to be accessed.
             */
            explicit VertexAccessor(VertexGeometry& geometry);
    
            /**
             * Constant VertexAccessor constructor.
             * @param geometry VertexGeometry that contains the vertices to be accessed.
             */
            explicit VertexAccessor(const VertexGeometry& geometry);

            /**
             * @param index Index of the requested vertex.
             * @return Constant data of vertex at given index.
             */
            const void* GetVertex(UInt32 index) const;

            /**
            * @param index Index of the requested vertex.
            * @return Mutable data of vertex at given index, or 0 if VertexGeometry is immutable.
            */
            void* GetMutableVertex(UInt32 index);

            /**
            * @return The number of vertices within the geometry.
            */
            UInt32 GetVertexCount() const;

        private:
            VertexGeometry::VertexArrayResource m_vertexResource;
            UInt32 m_stride;
            UInt32 m_count;
    };


    /**
     * VertexGeometry::VertexElementFormat helper class.
     */
    class ElementFormatAccessor {
        public:
            /**
             * ElementFormatAccessor constructor.
             * @param vertexGeometry VertexGeometry that contains the element formats to be accessed.
             */
            explicit ElementFormatAccessor(const VertexGeometry& vertexGeometry);

            /**
             *  Gets the according VertexElementFormat if the specified usage exists at the specified usage index.
             *  @param usage VertexGeometry::VertexUsage to look for in the vertex element format array.
             *  @param index UsageIndex to check if usage exists at.
             *  @return The according vertex element format if one exists with Usage and UsageIndex
             *          being equal to usage and index.
             */
            const VertexGeometry::VertexElementFormat* GetElementFormatByUsageAndIndex(VertexGeometry::VertexUsage usage, UInt8 index) const;

        private:
            VertexGeometry::FormatArrayResource m_resource;
            UInt32 m_count;
    };



    /**
     * Safely access vertex indices in a VertexGeometry.
     */
    class IndexAccessor {
        public:
            /**
             * Default constructor.
             */
            IndexAccessor();

            /**
             * Mutable IndexAccessor constructor.
             * @param geometry VertexGeometry that contains the vertices to be accessed.
             */
            explicit IndexAccessor(VertexGeometry& geometry);
            /**
             * Constant IndexAccessor constructor.
             * @param geometry VertexGeometry that contains the vertices to be accessed.
             */
            explicit IndexAccessor(const VertexGeometry& geometry);

            /**
             * @param position Position of the requested index.
             * @return Constant index at the given position.
             */
            UInt32 GetIndex(UInt32 position) const;
            
            /**
             * @param position Position of the index to set.
             * @param index Index value.
             * @return true if index is set, false otherwise (immutable VertexGeometry or position is out of range).
             */
            bool SetIndex(UInt32 position, UInt32 index);

            /**
             * @return Number of indices that can be retrieved.
             */
            UInt32 GetCount() const;

        private:
            Internal::ResourceData m_indexResource;
            UInt32 m_count;
            VertexGeometry::BufferType m_bufferType;
    };

    inline UInt32 IndexAccessor::GetCount() const
    {
        return m_count;
    }

/**  @} */ // end of Core3D

} // namespace Candera

#endif    // CANDERA_VERTEXGEOMETRY_H

