//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "VertexGeometryAccessors.h"

namespace Candera {

    ElementFormatAccessor::ElementFormatAccessor(const VertexGeometry& vertexGeometry):
        m_resource(vertexGeometry.GetFormatArrayResourceHandle()),
        m_count(vertexGeometry.GetVertexElementCount())
    {

    }

    const VertexGeometry::VertexElementFormat* ElementFormatAccessor::GetElementFormatByUsageAndIndex(VertexGeometry::VertexUsage usage, UInt8 index) const
    {
        const VertexGeometry::VertexElementFormat* formatArray = m_resource.GetData();
        if (formatArray == 0) {
            return 0;
        }

        for (UInt16 i = 0; i < m_count; ++i) {
            if ((formatArray[i].Usage == usage) &&
                (formatArray[i].UsageIndex == index)) {
                return formatArray + i;
            }
        }

        return 0;
    }

    VertexAccessor::VertexAccessor(VertexGeometry& geometry):
        m_vertexResource(geometry.GetVertexArrayResourceHandle()),
        m_stride(geometry.GetVertexStride()),
        m_count(geometry.GetVertexCount())
    {
    }
    VertexAccessor::VertexAccessor(const VertexGeometry& geometry):
        m_vertexResource(geometry.GetVertexArrayResourceHandle()),
        m_stride(geometry.GetVertexStride()),
        m_count(geometry.GetVertexCount())
    {
    }
    const void* VertexAccessor::GetVertex(UInt32 index) const
    {
        return ((index < m_count) && (m_vertexResource.GetData() != 0)) ? FeatStd::Internal::PointerAdd(m_vertexResource.GetData(), static_cast<OffsetType>(m_stride * index)) : 0;
    }
    void* VertexAccessor::GetMutableVertex(UInt32 index)
    {
        return ((index < m_count) && (m_vertexResource.GetMutableData() != 0)) ? FeatStd::Internal::PointerAdd(m_vertexResource.GetMutableData(), static_cast<OffsetType>(m_stride * index)) : 0;
    }

    UInt32 VertexAccessor::GetVertexCount() const
    {
        return m_count;
    }

    IndexAccessor::IndexAccessor():
        m_indexResource(ResourceDataHandle::InvalidHandle()),
        m_count(0),
        m_bufferType(VertexGeometry::ArrayBuffer)
    {
    }

    IndexAccessor::IndexAccessor(VertexGeometry& geometry):
        m_indexResource(geometry.GetIndexArrayResourceHandle()),
        m_count(geometry.GetIndexCount()),
        m_bufferType(geometry.GetBufferType())
    {
        if (m_bufferType == VertexGeometry::ArrayBuffer) {
            m_count = geometry.GetVertexCount();
        }
    }
    IndexAccessor::IndexAccessor(const VertexGeometry& geometry):
        m_indexResource(geometry.GetIndexArrayResourceHandle()),
        m_count(geometry.GetIndexCount()),
        m_bufferType(geometry.GetBufferType())
    {
        if (m_bufferType == VertexGeometry::ArrayBuffer) {
            m_count = geometry.GetVertexCount();
        }
    }

    UInt32 IndexAccessor::GetIndex(UInt32 position) const
    {
        UInt32 result = position;
        if (position < m_count) {
            switch (m_bufferType) {
                case Candera::VertexGeometry::UInt16IndexedArrayBuffer:
                    result = *(FeatStd::Internal::PointerToPointer<const UInt16*>(m_indexResource.GetData()) + position);
                    break;
                case Candera::VertexGeometry::UInt8IndexedArrayBuffer:
                    result = *(FeatStd::Internal::PointerToPointer<const UInt8*>(m_indexResource.GetData()) + position);
                    break;
                case Candera::VertexGeometry::UInt32IndexedArrayBuffer:
                    result = *(FeatStd::Internal::PointerToPointer<const UInt32*>(m_indexResource.GetData()) + position);
                    break;
                default:
                    break;
            }
        }

        return result;
    }

    bool IndexAccessor::SetIndex(UInt32 position, UInt32 index)
    {
        if (m_indexResource.GetMutableData() == 0) {
            return false;
        }

        switch (m_bufferType) {
            case Candera::VertexGeometry::UInt16IndexedArrayBuffer:
                *(FeatStd::Internal::PointerToPointer<UInt16*>(m_indexResource.GetMutableData()) + position) = static_cast<UInt16>(index); break;
            case Candera::VertexGeometry::UInt8IndexedArrayBuffer:
                *(FeatStd::Internal::PointerToPointer<UInt8*>(m_indexResource.GetMutableData()) + position) = static_cast<UInt8>(index); break;
            case Candera::VertexGeometry::UInt32IndexedArrayBuffer:
                *(FeatStd::Internal::PointerToPointer<UInt32*>(m_indexResource.GetMutableData()) + position) = index; break;
            default:
                return false;
        }

        return true;
    }

} // namespace Candera
