//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_VERTEXGEOMETRY_H)
#define CANDERA_VERTEXGEOMETRY_H

#include <Candera/Environment.h>
#include <Candera/EngineBase/Common/CanderaObject.h>
#include <Candera/EngineBase/Common/ResourceObject.h>
#include <Candera/System/MemoryManagement/MemoryManagement.h>

namespace Candera {

/**  @addtogroup Core3D
 *   @{
 */

/**
 *  @brief   The VertexGeometry holds all vertex data of a mesh. VertexGeometry may be used
 *           also for multiple VertexBuffer. The VertexGeometry holds also a full description of his
 *           own members. This makes it possible to map vertex data automated to shader attributes.
 */
class VertexGeometry : public CanderaObject
{
    FEATSTD_TYPEDEF_BASE(CanderaObject);

    public:
        /**
         *  Declaration of the usage of one vertex element.
         */
        enum VertexUsage
        {
            Position = 0x1,         ///< The position of a vertex in local space.
            PositionTransformed,    ///< The position of a vertex in world space.
            Normal,                 ///< The normal corresponding to one vertex/face.
            TextureCoordinate,      ///< The UV coordinates of one vertex.
            Color,                  ///< The color of one vertex.
            BlendWeight,            ///< The blend weight of one vertex.
            BlendIndex,             ///< The blend index of one vertex.
            PointSize,              ///< The point size of one vertex.
            Tangent,                ///< The tangent corresponding to one vertex.
            BiNormal,               ///< The bi-normal corresponding to one vertex.
            TesselationFactor,      ///< The tessellation factor corresponding to one vertex.
            Fog,
            Depth,
            Sample,
            Custom,                 ///< A custom usage identifier, use this for your specialized data.

            VertexUsageCount
        };

        /**
         *  To specify the data type of one vertex element.
         */
        enum VertexDataType
        {
            Float32_1       =  0,
            Float32_2       =  1,
            Float32_3       =  2,
            Float32_4       =  3,
            UInt8_1         =  4, ///< Widen to (value, 0, 0, 1)
            UInt8_1_N       =  5, ///< Bytes will be normalized (value/255.0F, 0, 0, 1)
            UInt8_2         =  6, ///< Widen to (value, value, 0, 1)
            UInt8_2_N       =  7, ///< Widen to normalized version (value/255, value/255, 0, 1)
            UInt8_3         =  8, ///< Widen to (value, value, value, 1)
            UInt8_3_N       =  9, ///< ...
            UInt8_4         = 10,
            UInt8_4_N       = 11,
            Int8_1          = 12,
            Int8_1_N        = 13, ///< Widen to normalized version (value/127.0F, 0, 0, 1)
            Int8_2          = 14,
            Int8_2_N        = 15,
            Int8_3          = 16,
            Int8_3_N        = 17,
            Int8_4          = 18,
            Int8_4_N        = 19,
            UInt16_1        = 20,
            UInt16_1_N      = 21, ///< widen to normalized version (value/65535.0F, 0, 0, 1)
            UInt16_2        = 22,
            UInt16_2_N      = 23,
            UInt16_3        = 24,
            UInt16_3_N      = 25,
            UInt16_4        = 26,
            UInt16_4_N      = 27,
            Int16_1         = 28,
            Int16_1_N       = 29, ///< widen to normalized version (value/32767.0F, 0, 0, 1)
            Int16_2         = 30,
            Int16_2_N       = 31,
            Int16_3         = 32,
            Int16_3_N       = 33,
            Int16_4         = 34,
            Int16_4_N       = 35,
            Float16_1       = 36,
            Float16_2       = 37,
            Float16_3       = 38,
            Float16_4       = 39,
            UInt32_1        = 40,
            UInt32_1_N      = 41, ///< widen to normalized version (value/4294967295.0F, 0, 0, 1)
            UInt32_2        = 42,
            UInt32_2_N      = 43,
            UInt32_3        = 44,
            UInt32_3_N      = 45,
            UInt32_4        = 46,
            UInt32_4_N      = 47,
            Int32_1         = 48,
            Int32_1_N       = 49, ///< widen to normalized version (value/2147483647.0F, 0, 0, 1)
            Int32_2         = 50,
            Int32_2_N       = 51,
            Int32_3         = 52,
            Int32_3_N       = 53,
            Int32_4         = 54,
            Int32_4_N       = 55,
            Float32_3x3     = 56, ///< Matrix with 3 lines and 3 columns.
            Float32_4x4     = 57, ///< Matrix with 4 lines and 4 columns.
            VertexDataTypeCount = 58  ///< Number of data types.
        };

        /**
         *  Define which memory pool to use.
         */
        enum MemoryPool
        {
            VideoMemory = 0,    ///< Use video memory.
            SystemMemory = 1    ///< Use system memory.
        };

        /**
         *  Define how the buffer data will be used.
         */
        enum BufferUsage
        {
            StaticWrite = 0,    ///< The buffer data will be specified once and used many times to draw.
            DynamicWrite = 1,   ///< The buffer data will be specified repeatedly and used many times to draw.
            InfrequentDraw = 2  ///< The buffer data will be specified once and used infrequently to draw.
        };

        enum BufferType
        {
            ArrayBuffer = 0,                ///< The VertexGeometry holds only an array of vertices.
            UInt16IndexedArrayBuffer = 1,   ///< The VertexGeometry holds an array of vertices and an index array describing a list of primitives.
            UInt8IndexedArrayBuffer = 2,    ///< The VertexGeometry holds an array of vertices and an index array describing a list of primitives.
            UInt32IndexedArrayBuffer = 3,   ///< The VertexGeometry holds an array of vertices and an index array describing a list of primitives.

            IndexedArrayBuffer = UInt16IndexedArrayBuffer   ///< Is a UInt16IndexedArrayBuffer.
        };

        /**
         *  The description of one Vertex Element.
         *  Example:
         *   ----------------------------------------------------------------
         *  | pos-x pos-y pos-z | u1 v1 | u2 v2 | normal-x normal-y normal-z |
         *   ----------------------------------------------------------------
         *
         *  This line is an example of one row of the "vertexGeometry" array in VertexGeometry::Data.
         *  The VertexElement array for a row like this would be:
         *
         *   VertexGeometry::VertexElement Mesh_Test_VertexElements[] =
         *   {
         *       {  0, VertexGeometry::Float32_3, VertexGeometry::Position, 0 },
         *       { 12, VertexGeometry::Float32_2, VertexGeometry::TextureCoordinate, 0 },
         *       { 20, VertexGeometry::Float32_2, VertexGeometry::TextureCoordinate, 1 },
         *       { 28, VertexGeometry::Float32_3, VertexGeometry::Normal, 0 }
         *   };
         *
         *  In this example the count of VertexElements in one row would be 4, which refers
         *  to the "vertexElementCount" member in the VertexGeometry::Data struct.
         *
         */
        struct VertexElementFormat
        {
            UInt16 Offset; // the offset in bytes from the beginning of one row
            VertexDataType Type; // the data type of the current VertexElement
            VertexUsage Usage; // the usage of the current VertexElement
            UInt8 UsageIndex; // the usage index of the current VertexElement in case of multiple
                              // VertexElements with the same usage
        };
        
        /**
         * VertexArrayResource is a ResourceObject that provides access to the vertex data.
         *
         * To create a resource handle, call VertexArrayResource::CreateHandle with the following parameters:
         * - data: Pointer to the vertex data.
         * - disposer: The function which defines the dispose method for the data.
         *    The function will be called on VertexGeometry::DisposeVertexData call or on VertexGeometry destruction.
         * - size: size of data in bytes.
         *
         * @see ResourceObject
         */
        typedef ResourceObject<const void> VertexArrayResource;
        typedef MemoryManagement::ArrayDisposer<const void*> VertexGeometryDisposer;
        typedef VertexArrayResource::DisposerFunction VertexGeometryDisposerFn;

        /**
         * FormatArrayResource is a ResourceObject that provides access to the VertexElementFormat data.
         *
         * To create a resource handle, call FormatArrayResource::CreateHandle with the following parameters:
         * - data: Pointer to the format data.
         * - disposer: The function which defines the dispose method for the data.
         *    The function will be called on VertexGeometry destruction.
         * - size: size of data in bytes.
         *
         * @see ResourceObject
         */
        typedef ResourceObject<const VertexElementFormat> FormatArrayResource;
        typedef MemoryManagement::ArrayDisposer<const VertexElementFormat*> VertexElementFormatDisposer;
        typedef FormatArrayResource::DisposerFunction VertexElementFormatDisposerFn;

        /**
         * IndexArrayResource is a ResourceObject that provides access to the index data.
         *
         * To create a resource handle, call IndexArrayResource::CreateHandle with the following parameters:
         * - data: Pointer to the index data.
         * - disposer: The function which defines the dispose method for the data.
         *    The function will be called on VertexGeometry::DisposeVertexData call or on VertexGeometry destruction.
         * - size: size of data in bytes.
         *
         * @see ResourceObject
         */
        typedef ResourceObject<const void> IndexArrayResource;
        typedef MemoryManagement::AdaptedArrayDisposer<const void*, const UInt16*> IndexBufferDisposer;
        typedef IndexArrayResource::DisposerFunction IndexBufferDisposerFn;

        /**
         *  Constructor for immutable const data.
         *  Takes data and a corresponding Disposer for the data.
         *  The method Disposer::Dispose will be called on destruction.
         *  If a Disposer is set to null the corresponding data will not be disposed on destruction.
         *  @param vertexArray                      The vertex geometry array.
         *  @param vertexGeometryDisposerFn         The function which defines the dispose method for the vertex geometry.
         *  @param vertexElementFormat              The vertex format.
         *  @param vertexElementFormatDisposerFn    The function which defines the dispose method for the vertex format.
         *  @param indexBuffer                      The index buffer.
         *  @param indexBufferDisposerFn            The function which defines the dispose method for the index buffer.
         *  @param vertexCount                      The count of vertices in vertexGeometry.
         *  @param vertexStride                     The stride of the vertexGeomtry.
         *  @param vertexElementCount               The count of vertex data elements in vertexGeometry.
         *  @param indexCount                       The count of indices in indexBuffer.
         *  @param memoryPool                       The memory pool of the vertexGeometry.
         *  @param bufferType                       The buffer type.
         *  @param bufferUsage                      The buffer usage.
         */
        VertexGeometry(const void* vertexArray,
                       VertexGeometryDisposerFn vertexGeometryDisposerFn,
                       const VertexGeometry::VertexElementFormat* vertexElementFormat,
                       VertexElementFormatDisposerFn vertexElementFormatDisposerFn,
                       const UInt16* indexBuffer,
                       IndexBufferDisposerFn indexBufferDisposerFn,
                       UInt32 vertexCount,
                       UInt16 vertexStride,
                       UInt16 vertexElementCount,
                       UInt32 indexCount,
                       MemoryPool memoryPool,
                       BufferType bufferType,
                       BufferUsage bufferUsage);

        /**
         *  Constructor for mutable data.
         *  Takes data and a corresponding Disposer for the data.
         *  The method Disposer::Dispose will be called on destruction.
         *  If a Disposer is set to null the corresponding data will not be disposed on destruction.
         *  @param vertexArray                      The vertex geometry array.
         *  @param vertexGeometryDisposerFn         The function which defines the dispose method for the vertex geometry.
         *  @param vertexElementFormat              The vertex format.
         *  @param vertexElementFormatDisposerFn    The function which defines the dispose method for the vertex format.
         *  @param indexBuffer                      The index buffer.
         *  @param indexBufferDisposerFn            The function which defines the dispose method for the index buffer.
         *  @param vertexCount                      The count of vertices in verteArray.
         *  @param vertexStride                     The stride of the vertexArray.
         *  @param vertexElementCount               The count of vertex data elements in vertexArray.
         *  @param indexCount                       The count of indices in indexBuffer.
         *  @param memoryPool                       The memory pool of the vertexArray.
         *  @param bufferType                       The buffer type.
         *  @param bufferUsage                      The buffer usage.
         */
        VertexGeometry(void* vertexArray,
                       VertexGeometryDisposerFn vertexGeometryDisposerFn,
                       VertexGeometry::VertexElementFormat* vertexElementFormat,
                       VertexElementFormatDisposerFn vertexElementFormatDisposerFn,
                       UInt16* indexBuffer,
                       IndexBufferDisposerFn indexBufferDisposerFn,
                       UInt32 vertexCount,
                       UInt16 vertexStride,
                       UInt16 vertexElementCount,
                       UInt32 indexCount,
                       MemoryPool memoryPool,
                       BufferType bufferType,
                       BufferUsage bufferUsage);

        /**
        *  Constructor that takes ResourceDataHandles for accessing vertex, index and format arrays.
        *  @param vertexArrayResourceHandle        ResourceDataHandle that references vertex data.
        *  @param indexArrayResourceHandle         ResourceDataHandle that references index data.
        *  @param formatArrayResourceHandle        ResourceDataHandle that references format data.
        *  @param vertexStride                     The stride of the vertexArray.
        *  @param memoryPool                       The memory pool of the vertexArray.
        *  @param bufferType                       The buffer type.
        *  @param bufferUsage                      The buffer usage.
        */
        VertexGeometry(const ResourceDataHandle& vertexArrayResourceHandle,
                       const ResourceDataHandle& indexArrayResourceHandle,
                       const ResourceDataHandle& formatArrayResourceHandle,
                       UInt16 vertexStride,
                       MemoryPool memoryPool,
                       BufferType bufferType,
                       BufferUsage bufferUsage);

        /**
         *  Destructor
         */
        virtual ~VertexGeometry() override;

        /**
         *  Retrieves if the VertexGeometry data can be changed or not.
         *  @return true if mutable, otherwise false.
         */
        CANDERA_DEPRECATED_3_0_2("Check if either vertex, index or format arrays are mutable: GetVertexArrayResourceHandle()/GetIndexArrayResourceHandle()/GetFormatArrayResourceHandle().m_isMutable",
            bool IsMutable() const);

        /**
         *  Retrieve the vertex array as a immutable const data.
         *  @return The vertex array.
         */
        CANDERA_DEPRECATED_3_0_2("Retrieve vertex array using VertexArrayResource(GetVertexArrayResourceHandle()).GetData()",
            const void* GetVertexArray() const);

        /**
         *  Retrieve the element format as a immutable const data.
         *  @return The element format.
         */
        CANDERA_DEPRECATED_3_0_2("Retrieve format array using FormatArrayResource(GetFormatArrayResourceHandle()).GetData()",
            const VertexElementFormat* GetVertexElementFormat() const);

        /**
         *  Retrieve the index buffer as a immutable const data.
         *  @return The index buffer.
         */
        CANDERA_DEPRECATED_3_0_2("Retrieve index array using IndexArrayResource(GetIndexArrayResourceHandle()).GetData()",
            const UInt16* GetIndexBuffer() const);

        /**
         *  Retrieves the vertex array if IsMutable() == true, otherwise NULL.
         *  @return  The vertex array if IsMutable() == true, otherwise NULL.
         */
        CANDERA_DEPRECATED_3_0_2("Retrieve mutable vertex array using VertexArrayResource(GetVertexArrayResourceHandle()).GetMutableData()",
            void* GetVertexArray());

        /**
         *  Retrieves the vertex format if IsMutable() == true, otherwise NULL.
         *  @return  The vertex format if IsMutable() == true, otherwise NULL.
         */
        CANDERA_DEPRECATED_3_0_2("Retrieve mutable format array using FormatArrayResource(GetFormatArrayResourceHandle()).GetMutableData()",
            VertexElementFormat* GetVertexElementFormat());

        /**
         *  Retrieves the index buffer if IsMutable() == true, otherwise NULL.
         *  @return  The index buffer if IsMutable() == true, otherwise NULL.
         */
        CANDERA_DEPRECATED_3_0_2("Retrieve mutable index array using IndexArrayResource(GetIndexArrayResourceHandle()).GetMutableData()",
            UInt16* GetIndexBuffer());

        /**
         * @return Constant VertexArrayResource handle.
         * Create a VertexArrayResource for accessing the constant vertex array data.
         */
        const ResourceDataHandle& GetVertexArrayResourceHandle() const { return m_vertexArrayResourceHandle; }
        
        /**
         * @return Mutable VertexArrayResource handle.
         * Create a VertexArrayResource for accessing the constant/mutable vertex array data.
         */
        ResourceDataHandle& GetVertexArrayResourceHandle() { return m_vertexArrayResourceHandle; }

        /**
         * @return Constant IndexArrayResource handle.
         * Create a IndexArrayResource for accessing the constant index array data.
         */
        const ResourceDataHandle& GetIndexArrayResourceHandle() const { return m_indexArrayResourceHandle; }
        
        /**
         * @return Mutable VertexArrayResource handle.
         * Create a IndexArrayResource for accessing the constant/mutable index array data.
         */
        ResourceDataHandle& GetIndexArrayResourceHandle() { return m_indexArrayResourceHandle; }

        /**
         * @return Constant FormatArrayResource handle.
         * Create a FormatArrayResource for accessing the constant VertexElementFormat array data.
         */
        const ResourceDataHandle& GetFormatArrayResourceHandle() const { return m_formatArrayResourceHandle; }

        /**
         * @return Mutable FormatArrayResource handle.
         * Create a FormatArrayResource for accessing the constant/mutable VertexElementFormat array data.
         */
        ResourceDataHandle& GetFormatArrayResourceHandle() { return m_formatArrayResourceHandle; }

        /**
         *  Gets the highest usage index stored in the vertex element format array.
         *  @return The highest usage index stored in the vertex element format array.
         */
        UInt8 GetMaxUsageIndex() const;

        /**
         *  Retrieve the vertex element size.
         *  @param vertexDataType The vertex data type.
         *  @return The vertex element size.
         */
        static UInt8 GetVertexElementSize(VertexDataType vertexDataType);

        /**
         *  Retrieve whether the vertex element is normalized or not.
         *  @param vertexDataType The vertex data type.
         *  @return true if the element is normalized, otherwise false.
         */
        static bool IsVertexElementNormalized(VertexDataType vertexDataType);

        /**
         *  Retrieve the vertex count.
         *  @return The vertex count.
         */
        UInt32 GetVertexCount() const { return m_vertexArrayResourceHandle.m_size / m_vertexStride; }

        /**
         *  Retrieve the vertex stride.
         *  @return The vertex stride.
         */
        UInt16 GetVertexStride() const { return m_vertexStride; }

        /**
         *  Retrieve the vertex element count.
         *  @return The vertex element count.
         */
        UInt16 GetVertexElementCount() const { return static_cast<UInt16>(m_formatArrayResourceHandle.m_size / sizeof(VertexElementFormat)); }

        /**
         *  Retrieve the memory pool.
         *  @return The memory pool.
         */
        VertexGeometry::MemoryPool GetMemoryPool() const { return m_memoryPool; }

        /**
         *  Retrieve the buffer type.
         *  @return The buffer type.
         */
        VertexGeometry::BufferType GetBufferType() const { return m_bufferType; }

        /**
         *  Retrieve the index count.
         *  @return The index count.
         */
        UInt32 GetIndexCount() const;

        /**
         *  Retrieve the buffer usage.
         *  @return The buffer usage.
         */
        VertexGeometry::BufferUsage GetBufferUsage() const { return m_bufferUsage; }

        /**
         *  Dispose a Vertex Data object.
         */
        void DisposeVertexData();

    private:
        UInt16 m_vertexStride; //The stride of a vertexGeometry row in bytes.
        VertexGeometry::MemoryPool m_memoryPool; //Specifies the pool in memory.
        VertexGeometry::BufferType m_bufferType; //Specifies the type of the VertexGeometry.
        VertexGeometry::BufferUsage m_bufferUsage; //Gives a hint how the buffer data will be used.
        ResourceDataHandle m_vertexArrayResourceHandle;
        ResourceDataHandle m_indexArrayResourceHandle;
        ResourceDataHandle m_formatArrayResourceHandle;
};

/**  @} */ // end of Core3D

} // namespace Candera

#endif    // CANDERA_VERTEXGEOMETRY_H

