//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_VERTEXBUFFER_H)
#define CANDERA_VERTEXBUFFER_H

#include <Candera/EngineBase/Common/InstanceId.h>
#include <Candera/Engine3D/Core/DeviceObject.h>
#include <Candera/Engine3D/Core/VertexGeometryAccessors.h>
#include <Candera/System/MemoryManagement/SharedPointer.h>
#include <Candera/System/MemoryManagement/Disposer.h>
#include <Candera/System/Container/ForwardIterator.h>
#include <FeatStd/Diagnostics/Measurable.h>

namespace Candera {
    /**  @addtogroup Core3D
     *   @{
     */

    class VertexGeometry;
    class PointSprite;
    class Billboard;
    class Billboard2D;
    class RenderDevice;


    /**
     *  @brief   The VertexBuffer encapsulates the attributes of an uploaded VertexGeometry.
     *           It holds the actual handles to the allocated and filled VRAM memory.
     *           Before is called the first time a VertexGeometry must be set, this will be upload to VRAM.
     *           The VertexBuffer implements manual reference counting to prevent multiple uploads into VRAM,
     *           the actual unload only takes place if the reference count gets 0.
     *           This enables the VertexBuffer to be shareable between different Meshes without
     *           multiple uploads into VRAM.
     *
     */
    class VertexBuffer: public DeviceObject, public FeatStd::Diagnostics::Measurable {
            FEATSTD_TYPEDEF_BASE(DeviceObject);
            struct InternalIterator;

        public:
            FEATSTD_TYPEDEF_SHARED_POINTER(VertexBuffer);

            typedef MemoryManagement::Disposer<const VertexGeometry*> VertexGeometryDisposer;
            typedef VertexGeometryDisposer::DisposerFunction VertexGeometryDisposerFn;
            /**
             *  The primitive types
             */
            enum PrimitiveType {
                Triangles,      ///< Triangle primitive, vertex geometry represents triplets of vertices forming triangles:   123456 -> 123, 456.
                TriangleStrip,  ///< Triangle primitive, vertex geometry represents triplets of vertices forming triangles:   123456 -> 123, 234, 345, 456.
                TriangleFan,    ///< Triangle primitive, vertex geometry represents triplets of vertices forming triangles:   123456 -> 123, 134, 145, 156.
                Lines,          ///< Line primitive, vertex geometry represents pairs of vertices forming lines:              123456 -> 12, 34, 56.
                LineStrip,      ///< Line primitive, vertex geometry represents list of connected vertices forming a line:    123456 -> 12, 23, 34, 45, 56.
                LineLoop,       ///< Line primitive, vertex geometry represents a circular list of connected vertices:        123456 -> 12, 23, 34, 45, 56, 61.
                Points          ///< Point primitive, vertex geometry represents single dots:                                 123456 -> 1, 2, 3, 4, 5, 6.
            };

            struct Triangle {
                UInt16 index[3];    /*!< Indices to triangle vertices, as occurred in memory. */
            };

            /**
             * Primitive definition.
             * Triangles, triangle fans and triangle strips use all three values.
             * Lines, line strips and line loops use only the first two values.
             * Points use only the first value.
             */
            struct Primitive {
                UInt32 m_index[3];
            };

            /**
             * ForwardIterator over Primitives.
             */
            typedef ForwardIterator<InternalIterator> PrimitiveIterator;

            /**
             *  Creates a VertexBuffer object with default values.
             *  @return Shared Pointer to the created VertexBuffer object.
             */
            FEATSTD_SHARED_POINTER_CREATE_DECLARATION();

            /**
             *  Destructor
             */
            virtual ~VertexBuffer() override;

            /**
             *  Sets the VertexBuffer as active in the global state
             *  @return Whether vertex buffer could be activated in the render device (true) or not(false.
             */
            bool Activate();

            /**
             *  Updates data within the VertexArray of an already uploaded VertexBuffer directly in VRAM.
             *  The data is taken from the VertexArray of the attached VertexGeometry in the VertexBuffer.
             *  @param index     The starting index (beginning from 0) of the VertexArray.
             *  @param count     The count of vertices to update.
             *  @return          Whether data in VRAM has been updated(true) or not(false).
             *                   Also fails if vertices are not in VRAM.
             */
            bool Update(UInt32 index, UInt32 count);

            /**
             *  Updates data within the VertexArray of an already uploaded VertexBuffer directly in VRAM.
             *  @param offset    The offset at which the update should start.
             *  @param size      The size in bytes of the data.
             *  @param data      The source data to update in the already uploaded VRAM VertexBuffer.
             *  @return          Whether data in VRAM has been updated(true) or not(false).
             *                   Also fails if vertices are not in VRAM.
             */
            bool Update(Int offset, Int size, const UInt8* data);

            /**
             *  Retrieve vertex array handle of graphic device, associated with the active resource pool.
             *  @return the vertex array handle of graphic device associated with the pool.
             */
            Handle GetVertexArrayMemoryHandle() const;

            /**
             *  Retrieve index buffer handle of graphic device, associated with the active context resource pool.
             *  @return the index buffer handle of graphic device associated with the pool.
             */
            Handle GetIndexBufferMemoryHandle() const;

            /**
             *  Sets the vertex geometry which will be uploaded to VRAM and the corresponding Disposer for the vertex geometry.
             *  This needs to be set before Upload can be called.
             *  @param vertexGeometry     The vertex geometry.
             *  @param disposerFn         The function which defines the dispose method for the vertex geometry.
             *                            If the Disposer is set to null the vertex geometry will not be disposed.
             *  @return                   True if vertex geometry was set. False if it's already uploaded.
             */
            bool SetVertexGeometry(VertexGeometry* vertexGeometry, VertexGeometryDisposerFn disposerFn = 0);

            /**
             *  Retrieves the vertex geometry.
             *  @return A pointer to the vertex geometry.
             */
            const VertexGeometry* GetVertexGeometry() const {
                return m_vertexGeometry;
            }
            VertexGeometry* GetVertexGeometry()
            {
                return m_vertexGeometry;
            }

            /**
             *  Retrieves the VertexGeometry if it is mutable (VertexGeometry::IsMutable() == true), otherwise NULL.
             *  @return  The VertexGeometry if it is mutable (VertexGeometry::IsMutable() == true), otherwise NULL.
             */
            CANDERA_DEPRECATED_3_1_1("Whether resources are mutable can be retrieved from the resource handle. Use GetVertexGeometry.",
                VertexGeometry* GetMutableVertexGeometry());

            /**
             *  Sets the vertex primitive type. @see PrimitiveType.
             *  This value defines how the vertices stored in VertexGeometry are connected and therefore how the primitive is assembled.
             *  @param primitiveType specifies the primitive type of this VertexBuffer.
             */
            void SetPrimitiveType(PrimitiveType primitiveType) {
                m_primitiveType = primitiveType;
            }

            /**
             *  Retrieves the vertex primitive type. @see PrimitiveType.
             *  @return The vertex primitive type.
             */
            PrimitiveType GetPrimitiveType() const {
                return m_primitiveType;
            }

            /**
             *  Renders the VertexGeometry of this VertexBuffer.
             */
            void Render();

            /**
             *  Overrides Measurable
             *  @return Size of vertex buffer + Size of index buffer from a vertex geometry in bytes.
             */
            virtual UInt GetSize() const override;

            /**
             * Retrieves a ForwardIterator that points at the first primitive. Primitive type can be checked by calling 
             *  GetPrimitiveType. For a complete list of supported primitives, see VertexBuffer::Primitive.
             * @return ForwardIterator over VertexBuffer primitives.
             * @see VertexBuffer::Primitive
             */
            PrimitiveIterator GetPrimitiveIterator() const;

            /**
             *  Iteration helper: Returns the first triangle in the vertex buffer. Resets current triangle to the first one.
             *  Note: If triangles shall be stored in a data structure, you have to instantiate the triangle struct accordingly,
             *        and copy the pointers contents.
             *  @return The first triangle in the vertex buffer or 0 (null) if no triangles exist. This also occurs if the vertex buffer holds lines or points.
             */
            CANDERA_DEPRECATED_3_1_1("Please iterate over triangles and primitives using iterator retrieved with GetPrimitiveIterator()",
            const Triangle* GetFirstTriangle());

            /**
             *  Iteration helper: Returns the next triangle in the vertex buffer.
             *  Assumes a previous call to GetFirstTriangle().
             *  Note: If triangles shall be stored in a data structure, you have to instantiate the triangle struct accordingly,
             *        and copy the pointers contents.
             *  @return The next triangle in the vertex buffer or 0 (null) if no triangles exist, the last triangle is already passed, or the first
             *          triangle is not set.
             */
            CANDERA_DEPRECATED_3_1_1("Please iterate over triangles and primitives using iterator retrieved with GetPrimitiveIterator().",
            const Triangle* GetNextTriangle());

            /**
             * Returns an immutable const void* to the vertex in the vertex array denoted by index.
             * The pointer is computed from GetVertexGeometry()->GetVertexArray + index * GetVertexGeometry()->GetVertexStride(),
             * as long as index is smaller than the amount of vertices, retrievable by GetVertexGeometry()->GetVertexCount().
             * Iteration over all vertices can be done as follows:
             * \code
             * for(Uint16 i = 0; i < GetVertexGeometry()->GetVertexCount(); i++) {
             *    const void* vertex = GetVertex(i);
             * }
             * \endcode
             *
             * @param index The index of the vertex to address. Has to be smaller than GetVertexGeometry()->GetVertexCount().
             * @return Pointer to the addressed vertex in the vertex array, if index is smaller than vertex count. 0 otherwise.
             */
            CANDERA_DEPRECATED_3_1_1("Retrieve vertex data with VertexAccessor helper class",
            const void* GetVertex(UInt32 index) const);

            /**
             * Returns a mutable void* to the vertex in the vertex array denoted by index, if GetVertexGeometry() is mutable.
             * The pointer is computed from GetVertexGeometry()->GetVertexArray + index * GetVertexGeometry()->GetVertexStride(),
             * as long as index is smaller than the amount of vertices retrievable by GetVertexGeometry()->GetVertexCount().
             * Iteration over all vertices can be done as follows:
             * \code
             * for(Uint16 i = 0; i < GetVertexGeometry()->GetVertexCount(); i++) {
             *    void* vertex = GetVertex(i);
             * }
             * \endcode
             *
             * @param index The index of the vertex to address. Has to be smaller than GetVertexGeometry()->GetVertexCount().
             * @return Pointer to the addressed vertex in the vertex array, if index is smaller than vertex count and GetVertexGeometry() is mutable.
             *         0 otherwise.
             */
            CANDERA_DEPRECATED_3_1_1("Retrieve vertex data with VertexAccessor helper class",
            void* GetMutableVertex(UInt32 index));

            /**
             * Returns the Id of this vertex buffer instance.
             * @return The Id of this vertex buffer instance.
             */
            Int GetInstanceId() const { return m_instanceId; }

            /**
             * Set the first element of the vertex buffer to start drawing with.
             * @param elementStart  The index of the first element to start drawing with.
             */
            void SetElementStart(SizeType elementStart) { m_elementStart = static_cast<Int>(elementStart); }

            /**
             * Get the first element of the vertex buffer to start drawing with.
             * @return The first element of the vertex buffer to start drawing with.
             */
            SizeType GetElementStart() const { return (0 > m_elementStart) ? 0 : static_cast<SizeType>(m_elementStart); }

            /**
             * Set the number of elements to be drawn.
             * @param elementCount The number of elements to be drawn.
             */
            void SetElementCount(SizeType elementCount) { m_elementCount = static_cast<Int>(elementCount); }

            /**
             * Get the number of elements to be drawn.
             * @return The number of elements to be drawn.
             */
            SizeType GetElementCount() const;

        protected:

            /**
             *  overridden from DeviceObject
             *  @return Whether uploading vertex geometry to VRAM succeeded (true), stays in system memory(true) or failed(false).
             */
            virtual bool UploadInternal(LoadingHint loadingHint) override;
            /**
             *  overridden from DeviceObject
             *  @return Whether unloading vertex geometry from VRAM succeeded (true), is in system memory and doesn't have to be unloaded(true)
             *  or failed(false).
             */
            virtual bool UnloadInternal(LoadingHint loadingHint) override;

            // override DeviceObject::DisposeInternal
            virtual void DisposeInternal() override;

        private:
            friend class RenderDevice;

            Internal::InstanceId<VertexBuffer> m_instanceId;

            struct TriangleIterator {
                Triangle triangle;    //Current triangle;
                UInt32 currentIndex;  //Current index of first triangle vertex in the case iteration is done over array triangles, current index
                //in index buffer if iterated over indexed triangles.
            };

            VertexGeometry* m_vertexGeometry;
            Handle m_vertexArrayMemoryHandle[CANDERA_MAX_CONTEXT_COUNT];
            Handle m_indexBufferMemoryHandle[CANDERA_MAX_CONTEXT_COUNT];
            VertexGeometryDisposerFn m_vertexGeometryDisposerFn;
            PrimitiveType m_primitiveType;
            TriangleIterator m_triangleIterator;
            Int m_elementStart;
            Int m_elementCount;

            // Explicit private Constructor and Copy-Constructor, use Create() to create an instance of this object
            FEATSTD_MAKE_CLASS_UNCOPYABLE(VertexBuffer);
            VertexBuffer();

            /**
             *  Sets the index array video memory handle
             *  @param handle     The index array memory handle.
             */
            void SetIndexArrayMemoryHandle(Handle handle);
            /**
             *  Sets the vertex array video memory handle
             *  @param handle     The vertex array memory handle.
             */
            void SetVertexArrayMemoryHandle(Handle handle);
            void DisposeVertexGeometry();

            // Make this class manageable by MemoryManagement::SharedPointer
            CANDERA_SHARED_POINTER_DECLARATION();

            struct InternalIterator {
                typedef VertexBuffer::Primitive ValueType;
                typedef VertexBuffer::Primitive ReferenceType;
                typedef VertexBuffer::Primitive PointerType;

                VertexBuffer::Primitive operator*() const;
                bool operator!=(const InternalIterator& other) const;
                InternalIterator& operator++();

                IndexAccessor m_indexAccessor;
                UInt32 m_index;
                VertexBuffer::PrimitiveType m_primitiveType;
            };
    };
    /**  @} */ // end of Core3D

} // namespace Candera

#endif  // CANDERA_VERTEXBUFFER_H
