//########################################################################
// (C) Candera GmbH
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Candera GmbH.
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_TEXTURE_IMAGE_H)
#define CANDERA_TEXTURE_IMAGE_H

#include <Candera/Engine3D/Core/DeviceObject.h>
#include <Candera/System/MemoryManagement/SharedPointer.h>

namespace Candera {
/** @addtogroup Core3D
 *  @{
 */
    class Texture;
    class ImageSource3D;
    class ContextResourcePool;
    struct TextureStateCache;

/**
* @brief  Abstract base class for Texture images.A texture image mainly encapsulates
*         a texture handle in VRAM and abstracts from using different texture target types.
*         It Provides mechanisms for uploading, unloading
*         and activation of a texture handle, which are called by Candera::Texture.
*/
    class TextureImage : public DeviceObject
    {
        friend class RenderDevice;
        FEATSTD_TYPEDEF_BASE(DeviceObject);

    public:

        /**
         * Defines which memory pool is used.
         */
        enum TextureMemoryPool
        {
            VideoMemory = 0,            ///< Uploads the texture to VideoMemory. This is not supported by all platforms.
            ClientMemory = 2            ///< Attaches a Bitmap's data pointer to a texture handle. This is not supported by all platforms.
        };

        /**
         * Specifies Texture target type of this TextureImage.
         */
        enum TextureTargetType
        {
            Texture2D = 0,           ///< Texture represents a 2D Texture.
            TextureCubeMap = 1,      ///< Texture represents a CubeMap Texture.
            TextureExternalOes = 2   ///< Texture represents an external texture.
        };

        /**
         *  Retrieves texture handle of graphic device, associated with the active resource pool.
         *  @return the texture handle of graphic device associated with the active pool.
         *  @return Texture handle of graphic device.
         */
        virtual Handle GetVideoMemoryHandle() const = 0;

        /**
         *  Retrieve if MipMapping is enabled or not
         *  @return   true, if mipmapping is enabled and mipmap filters are evaluated.
         *  @return   false, if mipmapping is disabled and mipmap filters are ignored.
         */
        virtual bool IsMipMappingEnabled() const = 0;

        /**
         * Activates the current texture image to be used with a certain texture unit.
         * @param unit The texture unit to be used with the texture image.
         * @return   true, if activation of texture image on texture unit succeeded, false otherwise.
         */
        FEATSTD_LINT_NEXT_EXPRESSION(1735,"Default argument on overriding virtual functions matches this default argument. (See ProxyImage.h)")
        virtual bool Activate(UInt unit = 0);

        /**
         *  Retrieves ImageSource3D of this TextureImage. ImageSource3D contains videohandle and if mipmapping is enabled.
         *  @return ImageSource3D of this TextureImage.
         */
        virtual ImageSource3D* ToImageSource3D() = 0;

        /**
         * Retrieves the texture target type of this TextureImage. This controls
         * how the texture image is activated.
         * @return TextureTargetType of this TextureImage.
         */
        virtual TextureTargetType GetTextureTargetType() const = 0;

        FEATSTD_RTTI_DECLARATION();

        /**
         * Set StateCache.
         * @param stateCache The cache of this texture image.
         */
        void SetStateCache(TextureStateCache* stateCache) { m_stateCache = stateCache; }

        /**
         * Get StateCache.
         * @return The cache of this texture image.
         */
        TextureStateCache* GetStateCache() { return m_stateCache; }
        /**
         * Get StateCache.
         * @return The cache of this texture image.
         */
        const TextureStateCache* GetStateCache() const { return m_stateCache; }

        /**
         * Gets the memory pool which shall be used to store the image date for rendering.
         * This property may be ignored on RenderDevices that only support uploading to VideoRam.
         * @return The memory pool which shall be used to store the image date for rendering.
         */
        TextureMemoryPool GetTextureMemoryPool() const { return m_memoryPool; }

        /**
        * Sets the memory pool which shall be used to store the image date for rendering.
        * This property may be ignored on RenderDevices that only support uploading to VideoRam.
        * @param memoryPool The memory pool which shall be used to store the image date for rendering.
        */
        void SetTextureMemoryPool(TextureMemoryPool memoryPool) { m_memoryPool = memoryPool; }

    protected:
        TextureImage();

    private:
        TextureStateCache* m_stateCache;
        TextureMemoryPool m_memoryPool;

        // Make this class manageable by MemoryManagement::SharedPointer
        CANDERA_SHARED_POINTER_DECLARATION();
    };
/** @} */ // end of Core3D
}

#endif
