//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_TEXTURE_H)
#define CANDERA_TEXTURE_H

#include <Candera/EngineBase/Common/Bitmap.h>
#include <Candera/Engine3D/Core/Transformable.h>
#include <Candera/Engine3D/Core/Shader.h>
#include <Candera/Engine3D/Core/BitmapTextureImage.h>
#include <Candera/Engine3D/Core/TextureImage.h>

namespace Candera {

/**  @addtogroup Core3D
 *   @{
 */

/**
 *  @brief   Texture encapsulates a TextureImage and a set of attributes specifying how it is applied to a vertex's texture coordinate.
 *             Candera implements a sharing mechanism based on TextureImages. These hold the actual VRAM handle. Multiple textures
 *             can share one TextureImage object. The TextureImage manages it's upload to the VRAM. Explicit Texture objects must
 *             only be created if the attributes of the Texture are not shareable.
 *
 *             Attention: The Texture is only usable with an associated TextureImage.
 *
 *             Note: Devices which support only OpenGL ES 2 or below, may have limited or no support for NPOT textures.
 *             For example, such devices may not allow mipmap generation on NPOT textures, or may not support repeat modes other than ClampToEdge.
 *
 *             ES2 devices with the 'OES_texture_npot' extension, or devices with ES3 and above, have full support for NPOT textures.
 *             To ensure support for MipMapping and repeated wrapping on previous devices, the bitmap's width and height must be
 *             a power of 2 (n^2), like e.g. 2, 4, 8, 16 ,32, etc. However, height and width can be different, e.g.: 256x128.
 *
 *
 *
 */
class Texture : public Transformable
{
    FEATSTD_TYPEDEF_BASE(Transformable);

    public:
        FEATSTD_TYPEDEF_SHARED_POINTER(Texture);
        /**
         *  MinMagFilter defines filter type used for minification and magnification.
         */
        enum MinMagFilter
        {
            MinMagNearest = 0,  ///< Value of the texel that is nearest to the center of the pixel being textured.
            MinMagLinear = 1    ///< The weighted average of the four texels that are closest to the center of the pixel being textured.
        };

        /**
         *  MipMapFilter defines filter type used for mipmapping.
         */
        enum MipMapFilter
        {
            MipMapNone = 0,     ///< Disables MipMapping.
            MipMapNearest = 1,  ///< MipMapping uses nearest mipmap level.
            MipMapLinear = 2    ///< MipMapping uses bilinear mipmap interpolation of the two nearest mipmap levels.
        };

        /**
         *  WrapMode specifies how the texture is wrapped if u,v texture coordinates exceed the range [0..1].
         */
        enum WrapMode
        {
            Repeat = 0,         ///< Repeat the texture.
            ClampToEdge = 1,    ///< Clamp fetches to the edge of the texture.
            MirroredRepeat = 2  ///< Repeat the texture and mirror.
        };

        /**
         *  Creates a Texture object with default values.
         *  @return Shared Pointer to the created Texture object.
         */
        static Texture::SharedPointer Create();

        /**
         *  Destructor
         */
        virtual ~Texture() override;

        /**
        *  Create a copy of this material.
        *  @return   A Shared pointer to the copy.
        */
        Texture::SharedPointer Clone() const;

        /**
         *  Uploads the Texture, the texture itself doesn't upload. This methods triggers the associated TextureImage to be uploaded.
         *  @return Whether the associated TextureImage was uploaded(true) or failed(false) to upload.
         */
        bool Upload();

        /**
         *  Unloads the Texture, the texture itself doesn't unload. This methods triggered the associated TextureIamge to be unloaded.
         *  @return Whether the associated TextureImage was uploaded(true) or failed(false) to unload.
         */
        bool Unload();

        /**
         *  Activates this Texture object and binds it to texture unit given in render device.
         *  @param shader that receives the Texture's uniform settings.
         *  @param unit specifies the texture unit where this Texture gets bound to.
         *  @return True if activating and binding the TextureImage to unit succeeded. False otherwise.
         */
        bool Activate(const Shader& shader, UInt unit = 0);

        /**
         *  Set magnification filter. Default value is "MinMagLinear".
         *  @param filter The magnification filter to set. Can be MinMagLinear or MinMagNearest.
         */
        void SetMagnificationFilter(MinMagFilter filter) { m_magFilter = filter; }

        /**
         *  Retrieve the magnification filter. Default value is "MinMagLinear".
         *  @return The magnification filter.
         */
        MinMagFilter GetMagnificationFilter() const { return m_magFilter; }

        /**
         *  Set minification filter. Default value is "MinMagLinear".
         *  @param filter The minification filter to set. Can be MinMagLinear or MinMagNearest.
         */
        void SetMinificationFilter(MinMagFilter filter) { m_minFilter = filter; }

        /**
         *  Retrieve the minification filter. Default value is "MinMagLinear".
         *  @return The minification filter.
         */
        MinMagFilter GetMinificationFilter() const { return m_minFilter; }

        /**
         *  Set mipmap filter. Default value is "MipMapNone".
         *  @param filter The mipmap filter to set. Can be MipMapNone, MipMapNearest or MipMapLinear.
         */
        void SetMipMapFilter(MipMapFilter filter) { m_mipMapFilter = filter; }

        /**
         *  Retrieve the mipmap filter. Default value is "MipMapNone".
         *  @return The mipmap filter.
         */
        MipMapFilter GetMipMapFilter() const { return m_mipMapFilter; }

        /**
         *  Set the texture level of detail (lod) bias used to restrict mipmap selection.
         *
         *  The lod bias defines an offset from the calculated mipmap level. For example, if a texture should be sampled at mipmap level 2 and lodBias is 1, then the texture will be sampled at mipmap level 3 instead.
         *
         *  @param lodBias Defines the index where the mipmap chain starts. Default is 0.
         */
        void SetLodBias(UInt16 lodBias) { m_lodBias = lodBias; }

        /**
        *  Retrieve the largest (in terms of texture size) mipmap level that will be used for the texture, this can also be called mipmap offset.
        *  @return The largest mipmap level for this texture.
        */
        UInt16 GetLodBias() const { return m_lodBias; }

        /**
        *  Set the texture level of detail (lod) lower limit used to restrict mipmap selection.
        *
        *  The internally calculated lod value lambda used to select a mipmap level is clamped to the set limits,
        *  where lambda <= 0 is the base mipmap level, lambda == 1 the second mipmap level and so on. A value of lambda == 0.5 means that
        *  when mipmap filtering is set to linear, the first and second mipmap levels will be blended equally. Negative lambda values result in magnification of the base mipmap level.
        *
        *  @param minLod Lower end of the mipmap range to clamp to. Default is -1000.
        */
        void SetMinLod(Float minLod) { m_minLod = minLod; }

        /**
        *  Retrieve the lower limit for the level of detail value calculated internally for mipmapping.
        *  @return The lower limit for the level of detail value.
        */
        Float GetMinLod() const { return m_minLod; }

        /**
        *  Set the texture level of detail (lod) limits used to restrict mipmap selection.
        *
        *  The internally calculated lod value lambda used to select a mipmap level is clamped to the set limits,
        *  where lambda <= 0 is the base mipmap level, lambda == 1 the second mipmap level and so on. A value of lambda == 0.5 means that
        *  when mipmap filtering is set to linear, the first and second mipmap levels will be blended equally. Negative lambda values result in magnification of the base mipmap level.
        *
        *  @param maxLod Upper end of the mipmap range to clamp to. This value should be greater than or equal to minLOD. Default is 1000.
        */
        void SetMaxLod(Float maxLod) { m_maxLod = maxLod; }

        /**
         *  Retrieve the upper limit for the level of detail value calculated internally for mipmapping.
         *  @return The upper limit for the level of detail value.
         */
        Float GetMaxLod() const { return m_maxLod; }

        /**
         *  Set wrap mode for texture coordinate U (x-dimension). Default value is "Repeat".
         *  @param wrapMode Wrap mode for texture coordinate U to set. Can be Repeat,ClampToEdge or MirroredRepeat.
         */
        void SetWrapModeU(WrapMode wrapMode) { m_wrapModeU = wrapMode; }

        /**
         *  Retrieve the wrap mode for texture coordinate U (x-dimension). Default value is "Repeat".
         *  @return The wrap mode for texture coordinate U (x-dimension).
         */
        WrapMode GetWrapModeU() const { return m_wrapModeU; }

        /**
         *  Set wrap mode for texture coordinate V (y-dimension). Default value is "Repeat".
         *  @param wrapMode Wrap mode for texture coordinate V to set. Can be Repeat,ClampToEdge or MirroredRepeat.
         */
        void SetWrapModeV(WrapMode wrapMode) { m_wrapModeV = wrapMode; }

        /**
         *  Retrieve the wrap mode for texture coordinate V (y-dimension). Default value is "Repeat".
         *  @return the wrap mode for texture coordinate V (y-dimension).
         */
        WrapMode GetWrapModeV() const { return m_wrapModeV; }

        /**
         *  Set the associated TextureImage of this Texture object.
         *  @param textureImage The texture image to associate this texture with.
         */
        void SetTextureImage(const MemoryManagement::SharedPointer<TextureImage>& textureImage) { m_textureImage = textureImage; }

        /**
         *  Retrieve the associated TextureImage of this Texture object.
         *  @return The associated TextureImage of this Texture object.
         */
        const MemoryManagement::SharedPointer<TextureImage>& GetTextureImage() const { return m_textureImage; }

        /**
         *  Set maximum degree of anisotropy for anisotropic filtering. Default value is 1.0F.
         *  @param maxAnisotropy specifies the maximum degree of anisotropy to account for in texture filtering for this Texture object.
         *  The value of maxAnisotropy must be greater or equal 1.0F to be valid. If value is less, it will not be set.
         *  The final degree of anisotropy rendered is limited by the max detail of anisotropy supported by hardware, which can be retrieved with
         *  function RenderDevice::GetMaxAnisotropySupportedByDevice.
         *  Note: Anisotropy filtering is an optional OpenGL ES 2.0 Extension (GL_EXT_texture_filter_anisotropic).
         *  If not supported, the max anisotropy is limited with 1.0F, which means no anisotropy (isotropy).
         */
        void SetMaxAnisotropy(Float maxAnisotropy);

        /**
         *  Retrieve the maximum degree of anisotropy for anisotropic filtering. Default value is 1.0F.
         *  @return The maximum degree of anisotropy for anisotropic filtering.
         */
        Float GetMaxAnisotropy() const { return m_maxAnisotropy; }

        /**
         *  Set the properties of this texture to its default values.
         */
        void SetPropertiesToDefault();

        /**
         *  Copies the properties the given source texture to this texture.
         *  @param sourceTexture  The texture to copy the properties for this texture from.
         */
        void CopyProperties(const Texture& sourceTexture);

        FEATSTD_RTTI_DECLARATION();

    private:
        MemoryManagement::SharedPointer<TextureImage> m_textureImage; // The associated TextureImage of this texture.

        Float m_maxAnisotropy;       // Max. degree of anisotropy.
        MinMagFilter m_magFilter;    // Magnification filter.
        MinMagFilter m_minFilter;    // Minification filter.
        MipMapFilter m_mipMapFilter; // MipMap filter.
        WrapMode m_wrapModeU;        // Wrap mode for texture coordinate U.
        WrapMode m_wrapModeV;        // Wrap mode for texture coordinate V.
        Float m_minLod;              // MipMap minimum level of detail clamp, default = RenderDevice::c_defaultMinLod
        Float m_maxLod;              // MipMap maximum level of detail clamp, default = RenderDevice::c_defaultMaxLod
        UInt16 m_lodBias;            // MipMap base level offset, default = 0

        // protected because only ::Create() should be used to create an instance of this class
        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1704, Candera::Texture::Texture, CANDERA_LINT_REASON_INSTANCESOBTAINABLE)
        Texture();
        Texture(const Texture& texture);
        Texture& operator = (const Texture& texture);

        // Make this class manageable by MemoryManagement::SharedPointer
        CANDERA_SHARED_POINTER_DECLARATION();

};

/**  @} */ // end of Core3D

} // namespace Candera

#endif    // CANDERA_TEXTURE_H

