//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "Texture.h"
#include <Candera/Engine3D/Core/BitmapTextureImage.h>
#include <Candera/Engine3D/Core/Renderer.h>
#include <Candera/System/Diagnostics/Log.h>

namespace Candera {
    FEATSTD_RTTI_DEFINITION(Texture, Transformable)
    using namespace Diagnostics;

    FEATSTD_LOG_SET_REALM(LogRealm::CanderaEngine3D);

CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1938, Candera::Texture::Texture, "Constructor accesses global default constants on purpose.")
Texture::Texture() :
    Base(),
    m_textureImage(),
    // Any changes here should also be applied to SetPropertiesToDefault()
    m_maxAnisotropy(1.0F),
    m_magFilter(MinMagLinear),
    m_minFilter(MinMagLinear),
    m_mipMapFilter(MipMapNone),
    m_wrapModeU(Repeat),
    m_wrapModeV(Repeat),
    m_minLod(RenderDevice::c_defaultMinLod),
    m_maxLod(RenderDevice::c_defaultMaxLod),
    m_lodBias(0U)
{
}

Texture::Texture(const Texture& texture) :
    Base(texture),
    m_textureImage(texture.m_textureImage),
    m_maxAnisotropy(texture.m_maxAnisotropy),
    m_magFilter(texture.m_magFilter),
    m_minFilter(texture.m_minFilter),
    m_mipMapFilter(texture.m_mipMapFilter),
    m_wrapModeU(texture.m_wrapModeU),
    m_wrapModeV(texture.m_wrapModeV),
    m_minLod(texture.m_minLod),
    m_maxLod(texture.m_maxLod),
    m_lodBias(texture.m_lodBias)
{
    // No instructions.
}

Texture::SharedPointer Texture::Create()
{
    CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(429, CANDERA_LINT_REASON_SHAREDPOINTER)
    Texture* ptr = FEATSTD_NEW(Texture)();
    if (ptr == 0) {
        FEATSTD_LOG_ERROR("Texture create failed, out of memory.");
    }
    Texture::SharedPointer sharedPointer(ptr);
    return sharedPointer;
}

Texture::~Texture()
{
}

Texture::SharedPointer Texture::Clone() const
{
    return Texture::SharedPointer(FEATSTD_NEW(Texture)(*this));
}

bool Texture::Upload()
{
    if(m_textureImage == 0) {
        FEATSTD_LOG_ERROR("Texture upload failed, texture image == 0.");
        return false;
    }

    return m_textureImage->Upload();
}

bool Texture::Unload()
{
    if(m_textureImage == 0) {
        FEATSTD_LOG_ERROR("Texture unload failed, texture image == 0.");
        return false;
    }

    return m_textureImage->Unload();
}

bool Texture::Activate(const Shader& shader, UInt unit)
{
    // Only resolve shared pointer once.
    TextureImage* textureImage = m_textureImage.GetPointerToSharedInstance();

    if (textureImage == 0) {
        FEATSTD_LOG_ERROR("Texture activate failed, texture image == 0.");
        return false;
    }

    bool result = true;
    const bool texImgActOk = textureImage->Activate(unit);
    if (!texImgActOk) {
        FEATSTD_LOG_ERROR("Texture Activate, textureImage->Activate unit %d failed.", unit);
        result = false;
    }

    bool tmpResult = RenderDevice::ActivateTexture(*this, unit);
    if (!tmpResult) {
        FEATSTD_LOG_ERROR("Texture activate, RenderDevice::ActivateTexture failed.");
        result = false;
    }

    // Retrieve associated semantic to texture target type of texture image used.
    ShaderParamNames::UniformSemantic semantic;
    switch (m_textureImage->GetTextureTargetType()) {
        case TextureImage::Texture2D:
            semantic = static_cast<ShaderParamNames::UniformSemantic>(static_cast<UInt>(ShaderParamNames::Texture) + unit);
            break;

        case TextureImage::TextureCubeMap:
            semantic = static_cast<ShaderParamNames::UniformSemantic>(static_cast<UInt>(ShaderParamNames::CubeMapTexture) + unit);
            break;

        default:
            semantic = static_cast<ShaderParamNames::UniformSemantic>(static_cast<UInt>(ShaderParamNames::Texture) + unit);
            FEATSTD_LOG_ERROR("Texture target type not supported");
            break;
    }

    // Only set the uniform if it is available / used in the shader.
    Int uniformLocation;
        
    if (shader.GetUniformLocation(semantic, uniformLocation)) {
        tmpResult = Renderer::SetUniformInRenderDevice<Shader::Integer>(uniformLocation, &unit);
        if (!tmpResult) {
            FEATSTD_LOG_ERROR("Uniform %s could not be set during Texture activation.", ShaderParamNames::GetUniformName(semantic));
            result = false;
        }
    }

    return result;
}

void Texture::SetMaxAnisotropy(Float maxAnisotropy)
{
    if (maxAnisotropy >= 1.0F) {
        m_maxAnisotropy = maxAnisotropy;
    }
}

void Texture::SetPropertiesToDefault()
{
    m_textureImage = MemoryManagement::SharedPointer<TextureImage>(0);
    m_maxAnisotropy = 1.0F;
    m_magFilter = MinMagLinear;
    m_minFilter = MinMagLinear;
    m_mipMapFilter = MipMapNone;
    m_wrapModeU = Repeat;
    m_wrapModeV = Repeat;
    m_minLod = RenderDevice::c_defaultMinLod;
    m_maxLod = RenderDevice::c_defaultMaxLod;
    m_lodBias = 0U;
}

void Texture::CopyProperties(const Texture& sourceTexture)
{
    m_textureImage = sourceTexture.m_textureImage;
    m_maxAnisotropy = sourceTexture.m_maxAnisotropy;
    m_magFilter = sourceTexture.m_magFilter;
    m_minFilter = sourceTexture.m_minFilter;
    m_mipMapFilter = sourceTexture.m_mipMapFilter;
    m_wrapModeU = sourceTexture.m_wrapModeU;
    m_wrapModeV = sourceTexture.m_wrapModeV;
    m_minLod = sourceTexture.m_minLod;
    m_maxLod = sourceTexture.m_maxLod;
    m_lodBias = sourceTexture.m_lodBias;
}

} // namespace Candera
