//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef CANDERA_TEXTMESH_H
#define CANDERA_TEXTMESH_H

#include <Candera/EngineBase/Common/Color.h>
#include <Candera/Engine3D/Core/VertexBuffer.h>
#include <Candera/System/Mathematics/Vector2.h>

namespace Candera {

namespace Internal {

/**
 *  @brief TextMesh manages given text using a vertex buffer. It is intended to be used to display text in a
 *         debug scenario. For that reason it is simple, lightweight, and independent from the TextEngine.
 */

class TextMesh
{

public:
    /**
     *  Destructor
     */
    virtual ~TextMesh();

    /**
     *  Create a text mesh.
     *  @param glyphData   An array of Vector4 where its components represent (width, height, x-stride, y-offset).
     *  @param uvs         An array holding the corresponding (u,v) coordinates.
     *  @param fontHeight  The height of the font.
     *  @param maxLength   The maximum number of characters the text mesh (i.e. vertex buffer) can hold.
     *  @param hasColor    A flag to indicate if vertex colors should be used to colorize the text.
     *  @return  A pointer to the created TextMesh.
     */
    static TextMesh* Create(const Vector<Vector4>& glyphData, const Vector<Vector2>& uvs, Float fontHeight,
        SizeType maxLength, bool hasColor = true);

    /**
     *  Set the width of the line. Text exceeding the line width will be positioned in the next line.
     *  @param lineWidth  The width of the line to set.
     */
    void SetLineWidth(Float lineWidth) { m_lineWidth = lineWidth; }

    /**
     *  Clear the text and set the pen to (0,0).
     */
    void Clear();

    /**
     *  Set the pen to the given position. TextMesh coordinates use the OpenGL convention of (0,0) being
     *  the bottom left corner with ascending Y going towards the top.
     *  @param penPosition  A Vector2 representing the (x,y) position to place the pen.
     */
    void SetPenPosition(const Vector2& penPosition) { m_penPosition = penPosition; }

    /**
     *  Set the color of the pen.
     *  @param  The color for the pen to use.
     */
    void SetPenColor(const Color& color) { m_color = color; }

    /**
     *  Append the given text at the current position of the pen using the pen color (if configured).
     *  @param text  A pointer to the text to append.
     */
    bool AppendText(const Char* text);

    /**
     *  Append the given text at the given position using the pen color (if configured).
     *  @param text      A pointer to the text to append.
     *  @param position  The (x,y) position where to append the text at.
     */
    bool AppendText(const Char* text, Vector2& position);

    /**
     *  Get the vertex buffer holding the text mesh.
     *  @return  The vertex buffer holding the text mesh.
     */
    const VertexBuffer::SharedPointer& GetVertexBuffer() const { return m_vertexBuffer; }

    /**
     *  Update the vertex buffer in VRAM. After text has been appended,
     *  the vertex buffer needs to be updated to reflect the changes.
     */
    void Update();

private:
    TextMesh(const Vector<Vector4>& glyphData, const Vector<Vector2>& uvs, Float fontHeight, SizeType maxLength, bool hasColor);
    TextMesh();
    FEATSTD_MAKE_CLASS_UNCOPYABLE(TextMesh);


    void CreateVertexBuffer(SizeType maxGlyphCount, bool hasVertexColor);

    VertexBuffer::SharedPointer m_vertexBuffer;

    Vector<Vector4> m_glyphData;
    Vector<Vector2> m_uvs;
    Vector2 m_penPosition;
    Color m_color;
    Float m_fontHeight;
    Float m_lineWidth;
    Float m_whitespaceWidth;
    bool m_hasColor;
    bool m_isDirty;
};

}

}

#endif
