//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "TextMesh.h"

namespace Candera {

FEATSTD_LOG_SET_REALM(Diagnostics::LogRealm::CanderaEngine3D);

namespace Internal {

struct VertexUv {
    // Position
    Float x;
    Float y;
    // UV
    Float u;
    Float v;
};

static const VertexGeometry::VertexElementFormat VertexUvFormat[] = {
    { 0, VertexGeometry::Float32_2, VertexGeometry::Position, 0 },
    { 8, VertexGeometry::Float32_2, VertexGeometry::TextureCoordinate, 0 }
};

struct VertexUvColor {
    // Position
    Float x;
    Float y;
    // UV
    Float u;
    Float v;
    // Color
    Float r;
    Float g;
    Float b;
    Float a;
};

static const VertexGeometry::VertexElementFormat VertexUvColorFormat[] = {
    { 0, VertexGeometry::Float32_2, VertexGeometry::Position, 0 },
    { 8, VertexGeometry::Float32_2, VertexGeometry::TextureCoordinate, 0 },
    { 16, VertexGeometry::Float32_4, VertexGeometry::Color, 0 }
};

TextMesh* TextMesh::Create(const Vector<Vector4>& glyphData, const Vector<Vector2>& uvs, Float fontHeight,
    SizeType maxLength, bool hasColor)
{
    return CANDERA_NEW(TextMesh)(glyphData, uvs, fontHeight, maxLength, hasColor);
}

TextMesh::TextMesh(const Vector<Vector4>& glyphData, const Vector<Vector2>& uvs, Float fontHeight,
    SizeType maxLength, bool hasColor)
    :
    m_glyphData(glyphData),
    m_uvs(uvs),
    m_penPosition(Vector2(0.0F, 0.0F)),
    m_color(Color(1.0F, 1.0F, 1.0F, 0.8F)),
    m_fontHeight(fontHeight),
    m_lineWidth(Math::MaxFloat()),
    m_whitespaceWidth(8.0F),
    m_hasColor(hasColor),
    m_isDirty(false)
{
    FEATSTD_DEBUG_ASSERT((glyphData.Size() * 4) == uvs.Size());
    FEATSTD_DEBUG_ASSERT((maxLength * 4) <= FeatStd::Internal::Limits<UInt16>::Max());
    CreateVertexBuffer(maxLength, hasColor);
    Clear();
}

TextMesh::~TextMesh()
{
    if (m_vertexBuffer != 0) {
        FEATSTD_DEBUG_ASSERT(FeatStd::MemoryManagement::Internal::IsSoleReference(m_vertexBuffer));
        m_vertexBuffer = VertexBuffer::SharedPointer(0);
    }
}

void TextMesh::Clear()
{
    FEATSTD_DEBUG_ASSERT(m_vertexBuffer != 0);
    m_vertexBuffer->SetElementCount(0);
    m_penPosition.SetZero();
}

bool TextMesh::AppendText(const Char* text)
{
    return AppendText(text, m_penPosition);
}

bool TextMesh::AppendText(const Char* text, Vector2& position)
{
    FEATSTD_DEBUG_ASSERT(m_vertexBuffer != 0);
    FEATSTD_DEBUG_ASSERT(0 != m_vertexBuffer->GetVertexGeometry());
    if (0 == text)
    {
        return true;
    }

    const SizeType offset = m_vertexBuffer->GetElementCount() / 6;
    const SizeType maxLength = static_cast<SizeType>(m_vertexBuffer->GetVertexGeometry()->GetVertexCount() / 4);
    SizeType count = 0;

    VertexGeometry::VertexArrayResource vertexResource(m_vertexBuffer->GetVertexGeometry()->GetVertexArrayResourceHandle());
    Float* vertices = const_cast<Float*>(FeatStd::Internal::PointerToPointer<const Float*>(vertexResource.GetData()));

    const Vector<Vector2>& uvs = m_uvs;

    for (SizeType j = 0; (Char(0) != text[j]); ++j) {
        if ((offset + count) >= maxLength) {
            break;
        }

        if (text[j] == ' ') {
            position[0] += m_whitespaceWidth;
            continue;
        }

        SizeType c = static_cast<SizeType>(text[j]);
        if (c >= m_glyphData.Size()) {
            return false;
        }

        SizeType i = (count + offset) * 4;
        Float cWidth = m_glyphData[c].GetX();
        Float cHeight = m_glyphData[c].GetY();
        Float cStride = m_glyphData[c].GetZ();
        if (((position[0] + cStride) >= m_lineWidth) || (c == static_cast<SizeType>('\n')))
        {
            position[0] = 0.0F;
            position[1] -= m_fontHeight;

            if (c == static_cast<SizeType>('\n'))
            {
                continue;
            }
        }

        const Float x = position[0];
        const Float cy = position[1] - m_glyphData[c].GetW();
        const Float x1 = position[0] + cWidth;
        const Float cy1 = cy + cHeight;
        const SizeType c4 = c * 4;

        SizeType k = i * ((m_hasColor ? sizeof(VertexUvColor) : sizeof(VertexUv))) / 4;

        // 1st vertex
        // position
        vertices[k++] = x;
        vertices[k++] = cy1;
        // uv
        vertices[k++] = uvs[c4].GetX();
        vertices[k++] = uvs[c4].GetY();
        // color
        if (m_hasColor) {
            vertices[k++] = m_color.GetRed();
            vertices[k++] = m_color.GetGreen();
            vertices[k++] = m_color.GetBlue();
            vertices[k++] = m_color.GetAlpha();
        }

        // 2nd vertex
        // position
        vertices[k++] = x1;
        vertices[k++] = cy1;
        // uv
        vertices[k++] = uvs[c4+1].GetX();
        vertices[k++] = uvs[c4+1].GetY();
        // color
        if (m_hasColor) {
            vertices[k++] = m_color.GetRed();
            vertices[k++] = m_color.GetGreen();
            vertices[k++] = m_color.GetBlue();
            vertices[k++] = m_color.GetAlpha();
        }

        // 3rd vertex
        // position
        vertices[k++] = x1;
        vertices[k++] = cy;
        // uv
        vertices[k++] = uvs[c4 + 2].GetX();
        vertices[k++] = uvs[c4 + 2].GetY();
        // color
        if (m_hasColor) {
            vertices[k++] = m_color.GetRed();
            vertices[k++] = m_color.GetGreen();
            vertices[k++] = m_color.GetBlue();
            vertices[k++] = m_color.GetAlpha();
        }

        // 4th vertex
        // position
        vertices[k++] = x;
        vertices[k++] = cy;
        // uv
        vertices[k++] = uvs[c4 + 3].GetX();
        vertices[k++] = uvs[c4 + 3].GetY();
        // color
        if (m_hasColor) {
            vertices[k++] = m_color.GetRed();
            vertices[k++] = m_color.GetGreen();
            vertices[k++] = m_color.GetBlue();
            vertices[k++] = m_color.GetAlpha();
        }

        position[0] += cStride;
        count++;
    }

    m_vertexBuffer->SetElementCount(m_vertexBuffer->GetElementCount() + (count * 6));
    m_penPosition = position;
    m_isDirty = true;
    return true;
}

void TextMesh::Update()
{
    if (!m_vertexBuffer->IsUploaded()) {
        if (!m_vertexBuffer->Upload(DeviceObject::NoHint)) {
            FEATSTD_LOG_ERROR("Could not upload vertex buffer for renderer statistics overlay.");
        }
    }
    else {
        if (m_isDirty) {
            if (m_vertexBuffer->Update(0, static_cast<UInt32>((m_vertexBuffer->GetElementCount() / 6) * 4))) {
                m_isDirty = false;
            }
        }
    }
}

void TextMesh::CreateVertexBuffer(SizeType maxGlyphCount, bool hasVertexColor)
{
    SizeType vertexCount = maxGlyphCount * 4;
    if (vertexCount > FeatStd::Internal::Limits<UInt16>::Max()) {
        // Number of vertices must not exceed 16 bit indices.
        return;
    }

    SizeType vertexSize = hasVertexColor ? sizeof(VertexUvColor) : sizeof(VertexUv);
    Float* vertices = CANDERA_NEW_ARRAY(Float, (vertexCount * vertexSize / sizeof(Float)));
    if (0 == vertices) {
        return;
    }

    SizeType indexCount = maxGlyphCount * 6;
    UInt16* indexBuffer = CANDERA_NEW_ARRAY(UInt16, indexCount);
    if (0 == indexBuffer) {
        CANDERA_DELETE_ARRAY(vertices);
        return;
    }

    for (SizeType i = 0; i < maxGlyphCount; ++i) {
        const UInt16 vertexOffset = FeatStd::Internal::NumericConversion<UInt16>(i * 4);
        const SizeType idx = i * 6;

        indexBuffer[idx] = 0 + vertexOffset;
        indexBuffer[idx + 1] = 2 + vertexOffset;
        indexBuffer[idx + 2] = 1 + vertexOffset;

        indexBuffer[idx + 3] = 0 + vertexOffset;
        indexBuffer[idx + 4] = 3 + vertexOffset;
        indexBuffer[idx + 5] = 2 + vertexOffset;
    }

    UInt16 vertexElementCount = (static_cast<UInt16>(hasVertexColor ? sizeof(VertexUvColorFormat) : sizeof(VertexUvFormat)) /
        sizeof(VertexGeometry::VertexElementFormat));
    VertexGeometry* vertexGeometry = CANDERA_NEW(VertexGeometry)(vertices,
        MemoryManagement::AdaptedArrayDisposer<const void*, const Float*>::Dispose,
        hasVertexColor ? VertexUvColorFormat : VertexUvFormat, 0,
        indexBuffer, MemoryManagement::AdaptedArrayDisposer<const void*, const UInt16*>::Dispose,
        static_cast<UInt32>(vertexCount), static_cast<UInt16>(vertexSize), 
        vertexElementCount,
        static_cast<UInt32>(indexCount),
        VertexGeometry::VideoMemory,
        VertexGeometry::UInt16IndexedArrayBuffer,
        VertexGeometry::DynamicWrite);

    VertexBuffer::SharedPointer vertexBuffer = VertexBuffer::Create();
    // ignore return flag, because the vertex buffer was just created and cannot fail due to being uploaded already.
    static_cast<void>(vertexBuffer->SetVertexGeometry(vertexGeometry, VertexBuffer::VertexGeometryDisposer::Dispose));

    m_vertexBuffer = vertexBuffer;
}

}

}
